/*************************************************************************************************
// INTEL CONFIDENTIAL Copyright 2011-2018 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to the source code
// ("Material") are owned by Intel Corporation or its suppliers or licensors. Title to the Material
// remains with Intel Corporation or its suppliers and licensors. The Material contains trade secrets
// and proprietary and confidential information of Intel or its suppliers and licensors. The Material is
// protected by worldwide copyright and trade secret laws and treaty provisions. No part of the
// Material may be used, copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual property right is
// granted to or conferred upon you by disclosure or delivery of the Materials, either expressly, by
// implication, inducement, estoppel or otherwise. Any license under such intellectual property
// rights must be express and approved by Intel in writing.
//*************************************************************************************************/

#ifndef SENSOR_DEFINE__
#define SENSOR_DEFINE__

#include <devpkey.h>
#include <Ntstrsafe.h>
#include <acpitabl.h>
#include "sensor_internal.h"
#include "vcm_type.h"
#include "nvram_type.h"

#include "SsCommon.h"
#include "SsGpio.h"
#include "SsI2c.h"
#include "SsVcm.h"
#include "SsNvm.h"

#ifdef SKY_CAM
    #include "public.h"
    #include "BIOSInterface.h"
    #include "sensor_skycam.h"
    #include "IvcamDriverInterface.h"
    #include "controllogic_sensor_intf.h"
#endif
#ifdef OTM_SENSOR_CONTROL_LOGIC
   #include "public.h"
   #include "BIOSInterface.h"
   #include "controllogic_sensor_intf.h"
#endif

/*! "brief flis clock
 */
#define IPC_COMMAND_BUF_SIZE        4    /*! in UINT32*/
#define IPCMSG_FLIS_CLOCK           0xE6
#define OSC_CLK_OUT1                1
#define OSC_CLK_OUT2                2
#define CLOCK_ON_PENWELL            1    /*! for Penwell*/
#define CLOCK_ON                    7    /*! for Cloverview*/
#define CLOCK_OFF                   0
#define REG_TOK_MASK                0xfff0

/*! "brief embedded data type
 */
#define DATA_TYPE_EMBEDDED          0x12

#if !defined(CHT_SLE)
#define CAMERA_SENSOR_CONFIGURABLE
#endif

#if defined(CHT_SLE)
#define DSM_NOT_READY
#endif

typedef enum TRACE_LEVEL_DEFINE
{
    FLAG_LOG,
    FLAG_WARN,
    FLAG_ERROR,
} tag_sensor_trace_level;

/*! "brief power state
 */
typedef enum _POWER_STATUS_
{
    PWR_DOWN,
    PWR_UP,
    PWR_STANDBY
} PWR_STATE;

/*! "brief resolution ratio
 */
typedef enum _RES_RATIO_
{
    RATIO_4_3,
    RATIO_16_9,
    RATIO_3_2,
    RATIO_NUM
} RES_RATIO;

/*! "brief resolution setting
 */
typedef struct _REGS_SETTING__
{
    DATA_TOKE_TYPE Type;
    UINT16 Reg;
    UINT32 Data;
    UINT32 DataOptional;
} REGS_SETTING;

typedef struct _REGS_SETTING__EXT_
{
    DATA_TOKE_TYPE Type;
    ADDR_TOKE_TYPE AddrType;
    UINT32 Reg;
    UINT32 Data;
    UINT32 DataOptional;
} REGS_SETTING_EXT;

/*! "brief setting unit
 */
typedef struct _S_SETTING__
{
    ULONG Reg;
    ULONG Data;
} S_SETTING;

typedef enum sensor_test_mode_status_
{
    SENSOR_TESTMODE_DISABLE,
    SENSOR_TESTMODE_ENABLE,
    SENSOR_TESTMODE_NOP,
} SENSOR_TEST_MODE_STATUS;
#if defined(SKY_CAM) || defined(OTM_SENSOR_CONTROL_LOGIC)
#define CONTROLLOGIC_SYMBOL_LINK_NAME_SIZE      100
#endif

/*! "brief resolution extend
 */
#define RES_EXTEND_0   0
#define RES_EXTEND_1   1
#define RES_EXTEND_2   2
#define RES_EXTEND_3   3
#define RES_EXTEND_MAX 4
#define RES_EXTEND_VTS RES_EXTEND_0
#define RES_EXTEND_HTS RES_EXTEND_1

/*! "brief sensor resolution data structure
 */
typedef struct _SENSOR_RESOLUTION_ 
{
#ifndef SKY_CAM
    ULONG Width;                                   /*!< Mandatory. Sensor resolution width. */
    ULONG Height;                                  /*!< Mandatory. Sensor resolution height. */
    LONG FPS;                                      /*!< Mandatory. Sensor resolution frame number per second. */
    ULONG Binning;                                 /*!< Mandatory. Sensor binning mode. */
    INT32 MipiBps;                                 /*!< Mandatory. Sensor mipi data bps. */
    ULONG BayerOrder;                              /*!< Mandatory. Sensor resolution bayer order. */
    ULONG BayerOrder180Degree;                     /*!< Mandatory. Sensor resolution bayer order when image mirror/flip. */
    ULONG IsFullFov;                               /*!< Mandatory. Whether the sensor resolution is full FOV. */
    ULONG PrivateOpt;                              /*!< Optional. Sensor private configuration. */
    REGS_SETTING *regs;                            /*!< Mandatory. Sensor setting. */
    ULONG ExtendOpt[RES_EXTEND_MAX];               /*!< Optional. Sensor extend setting. */
    MIPI_TIMING Mipi;                              /*!< Optional. Sensor mipi timing debug. */
#else
    ULONG Width;                                   /*!< Mandatory. Sensor resolution width. */
    ULONG Height;                                  /*!< Mandatory. Sensor resolution height. */
    ULONG CropX;                                   /*!< Optional. Sensor resolution crop in width. */
    ULONG CropY;                                   /*!< Optional. Sensor resolution crop in height. */
    LONG FPS;                                      /*!< Mandatory. Sensor resolution frame number per second. */
    ULONG BayerOrder;                              /*!< Mandatory. Sensor resolution bayer order. */
    ULONG BayerOrder180Degree;                     /*!< Mandatory. Sensor resolution bayer order when image mirror/flip. */
    ULONG IsFullFov;                               /*!< Mandatory. Whether the sensor resolution is full FOV. */
#endif
} SENSOR_RESOLUTION;

/*! "brief SNSR_MIPI_CONFIG
 *       sensor mipi configration
 */
typedef struct _SNSR_INTERNAL_CONFIG
{
    ULONG MipiPort;         /*!< csi mipi port. */
    ULONG MipiLane;         /*!< mipi lanes number. */
    ULONG OutputFormat;     /*!< sensor output format */
    ULONG Type;             /*!< sensor type */
} SNSR_MIPI_CONFIG;

/*! "brief AE register values
 */
typedef struct
{
    UINT16 FLines;
    UINT16 Coarse;
    UINT16 AnalogGain;
    UINT16 DigitalGain;
    BOOLEAN FLinesValid;
    BOOLEAN CoarseValid;
    BOOLEAN AnalogGainValid;
    BOOLEAN DigitalGainValid;
} AERegValues;

/*! "brief firmware
 */
typedef struct{
    UINT8*       data;  /*!< pointer to the firmware data */
    ULONG        bytes; /*!< length in bytes of firmware data */
} FIRMWARE;

/*! "brief ae exposure data status
 */
typedef enum __ExpStatus
{
    EXPSTS_INVALID,
    EXPSTS_UPDATED,
    EXPSTS_FETCHED
} EXP_STATUS_EN;

/*! "brief ae exposure time data
 */
typedef struct __EXP_
{
    SNSR_EXPOSURE Exp;
    EXP_STATUS_EN ExpStatus;
    EXP_STATUS_EN GainStatus;
    EXP_STATUS_EN DGainStatus;
} EXP_DATA_ST;

/*! "brief exposure time table number
 */
#define EXP_TABLE_NUM 16


/*! "brief DEVICE_CONTEXT
 *       sensor device context
 */
typedef struct _DEVICE_CONTEXT
{
    WDFDEVICE               DeviceHandle;              /*! sensor device handle */
    PDEVICE_OBJECT          DeviceObject;              /*! sensor device objective */
    GUID                    Guid;                      /*! sensor guid */
    SNSR_VERSION_INFO       Ver;                       /*! sensor version */
    GPIO_CONTEXT            GpioDev;                   /*! sensor gpio context */
    I2C_CONTEXT             I2cDev;                    /*! sensor i2c context */
    VCM_CONTEXT             SsVcm;                     /*! sensor vcm context */
    NVM_CONTEXT             SsNvm;                     /*! sensor nvram context */
    LONG                    ReferenceCount;            /*! sensor reference count */
    HANDLE                  PowerOnEvent;              /*! sensor power on sync mutex event */
    KEVENT                  SettingEvent;              /*! sensor i2c access sync mutex event */
    PWR_STATE               PowerOnStatus;             /*! sensor power on status */
    ULONG                   Orientation;               /*! sensor orientation */
    INT32                   ResIndex;                  /*! sensor resolution index */
    SNSR_MODE               Mode;                      /*! sensor mode */
    BOOLEAN                 ModeEnableList[maxMode];   /*! sensor mode enable list */
    SNSR_INFO               Info;                      /*! sensor information */
    DRIVER_STAGE            DriverStage;               /*! sensor driver stage*/
    UINT8                   Binning;                   /*! sensor binning mode */
    INT32                   MipiBps;                   /*! sensor mipi data mbps */
    UINT16                  QueueCoarse;               /*! sensor ae queue coarse intigration */
    UINT16                  QueueAnalogGain;           /*! sensor ae queue analog gain */
    UINT16                  QueueAnalogGainDelay;      /*! sensor ae queue analog gain delay */
    UINT16                  QueueDigitalGain;          /*! sensor ae queue digital gain */
    UINT16                  QueueDigitalGainDelay;     /*! sensor ae queue digital gain delay */
    UINT16                  QueueFrameLengthLines;     /*! sensor ae queue frame length */
    AERegValues             AERegValueCache;           /*! sensor ae reg value cache */
    BOOLEAN                 Streaming;                 /*! sensor streaming status */
    BOOLEAN                 FirstExposure;             /*! sensor is first exposure time */
    DSDT_CAMERA_SETTINGS    DsdtSettings;              /*! sensor dsdt setting */
    UINT8                   GpioPwdActVal;             /*! sensor gpio power down active value */
    UINT16                  Vts;                       /*! sensor vts */
    UINT16                  MinVts;                    /*! sensor min vts */
    UINT16                  LastVts;                   /*! sensor last vts */
    UINT8                   BankId;                    /*! sensor bank id */
    ULONG                   MclkPort;                  /*! sensor m-clock port */
    ACPI_PLD_V2_BUFFER      Pld;                       /*! sensor pld */
    PLAT_INFO               PlatformInfo;              /*! sensor platform info */
    BOOLEAN                 IsFwLoad;                  /*! sensor is firmware loaded */
    BOOLEAN                 IsFirstTimeStream;         /*! sensor is first time streaming */
    BOOLEAN                 IsEmbeddedDataEnabled;     /*! sensor is embedded data enabled */
    BOOLEAN                 IsPdafDataEnabled;         /*! sensor is pdaf data  */
#if defined(CAMERA_SENSOR_CONFIGURABLE)
    BOOLEAN                 Key_IsRegCheck;            /*! sensor key reg check: false-disable, true-enable */
    UINT8                   Key_MipiLanes;             /*! sensor key mipi lanes: 1-1, 2-2, 4-4, 0-lanes depended on driver define */
    UINT8                   Key_MipiPort;              /*! sensor key mipi PORT 0/1/2/3  depended on HW connection */
    UINT8                   Key_MipiDataFormat;        /*! sensor key mipi data format RAW8/RAW10 depended on driver define */
    UINT32                  Key_MipiMbps;              /*! sensor key mipi data output speed depended on driver define */
#endif
#ifndef SKY_CAM
    SENSOR_INTERFACE        SensorInterface;           /*! sensor interface */
    DSDT2_CAMERA_SETTINGS   Dsdt2Settings;             /*! sensor dsdt 2 settings */
    UINT8                   Key_EnableEmbeddedData;    /*! sensor key embedded line: 0-disable, 1-enable, -1-default value */
    FIRMWARE                SensorFirmware;            /*! sensor firmware */
    LONG                    ExpEffectDelay;            /*! sensor exposure effect delay */
    LONG                    AGainEffectDelay;          /*! sensor exposure AG effect delay */
    LONG                    DGainEffectDelay;          /*! sensor exposure DG effect delay */
    KEVENT                  ExpEvent;                  /*! sensor exposure sync event */
    ULONG                   CurYmean;                  /*! sensor exposure current Y mean */
    LONG                    CurPixClkFreqHz;           /*! sensor current pixel clock frequency in hz */
    LONG                    DefaultHTS;                /*! sensor default hts */
    LONG                    DefaultVTS;                /*! sensor default vts */
    LONG                    DefaultDelay;              /*! sensor default delay */
    PIO_WORKITEM            WorkItemExposure;          /*! sensor exposure work item */
    KDPC                    DPCExposure;               /*! sensor exposure dpc */
    KTIMER                  TimerExposure;             /*! sensor exposure timer */
    LONG                    preTimeExp;                /*! sensor exposure pre time */
    LONG                    remainTimeExp;             /*! sensor exposure remain time */
    LONG                    N_Frame;                   /*! sensor exposure the Nth frame */
    LONG                    N_exp_id;                  /*! sensor exposure the Nth exp id */
    BOOLEAN                 IsThereNewExp;             /*! sensor exposure is there new exposure time */
    EXP_DATA_ST             ExpFetch;                  /*! sensor exposure effect fetch */
    EXP_DATA_ST             ExpTab[EXP_TABLE_NUM];     /*! sensor exposure effect tab */
#if defined(OTM_SENSOR_CONTROL_LOGIC)
    SENSOR_DRIVER_VIEW      SensorDriverView;          /*! sensor driver view */
    SNSR_SKU                SKU;                       /*! sensor sku */
    CONTROLLOGIC_SENSOR_INTERFACE* ControlLogicInterface; /*! sensor control logic interface */
    PVOID SensorNotificationEntry;                     /*! sensor notification entry */
    PDEVICE_OBJECT          ControlLogicDeviceObject;  /*! sensor control logic device object */
    CHAR                    ControlLogicSymbolLink[CONTROLLOGIC_SYMBOL_LINK_NAME_SIZE];/*! sensor symbol */
    CONTROLLOGIC_INFO       ControlLogicInfo;          /*! sensor control logic information */
    SENSOR_CONF             SensorConf;                /*! sensor sensor configration*/
#endif  
#else
    SENSOR_INTERFACE        SensorInterface;           /*! sensor interface */
    SENSOR_DRIVER_VIEW      SensorDriverView;          /*! sensor control logic driver view */
    SNSR_SKU                SKU;                       /*! sensor control sku */
    CONTROLLOGIC_SENSOR_INTERFACE* ControlLogicInterface;  /*! sensor control logic interface */
    PVOID SensorNotificationEntry;                     /*! sensor notification entry */
    PDEVICE_OBJECT          ControlLogicDeviceObject;  /*! sensor control logic device object */
    PSENSOR_CONTROL_LINK    IvcamSCLInterface;         /*! sensor ivcam scl interface */
    PDEVICE_OBJECT          IvcamSCLDeviceObject;      /*! sensor ivcam scl device object */
    CHAR                    ControlLogicSymbolLink[CONTROLLOGIC_SYMBOL_LINK_NAME_SIZE]; /*! sensor control logic symbol */
    CONTROLLOGIC_INFO       ControlLogicInfo;          /*! sensor control logic information */

    SENSOR_CONF             SensorConf;                /*! sensor configuration */
    SNSR_RESOLUTION_DEBUG   LatestResolution;          /*! sensor latest resolution */
    SNSR_EXPOSURE           LatestExposureTime;        /*! sensor latest exposure time */
#endif

    ULONG                   SysPowerAction;            /*! sensor system power action for GPIO resources*/

#ifdef NVMPARSER_LOGIC
#define IA_NVM_NUM_CHANNELS 4
    UINT8                   cie_coords_x;              /*! sensor nvram parser logic */
    UINT8                   cie_coords_y;
    INT16                   vcm_af_near;
    INT16                   vcm_af_far;
    INT16                   vcm_af_start;
    INT16                   vcm_af_end;
    UINT16                  awb_sensitivities[IA_NVM_NUM_CHANNELS];
#endif
#ifdef PIXTER_DEFINE
    MIPI_TIMING2            Mipi;                      /*! sensor pixter define */
    PIXTER_VC_SETTING       vc_setting[4];
    PIXTER_SETTING          *settings;
    UINT32                  cur_setting;
    UINT32                  setting_num;
    PIXTER_FPS              dbg_fps;
#endif
    BOOLEAN                 DumpNVMData;

    BOOLEAN                 IsLowerFpsNeeded; /*! sensor need to use the setting with lower fps. */
    ULONG                   FrameLines;
    //This flag is used to perform VHDR specific workaround for OV5693
    BOOLEAN                 IsVhdrEnabled;

    SNSR_CAPS               SensorCaps;
    SNSR_MIPI_CONFIG        SensorMipiConfig;
    ULONG_PTR               IsTestModeEnabled;
    UNICODE_STRING          DriverPath;
} DEVICE_CONTEXT, *PDEVICE_CONTEXT;

/*! "brief sensor private declare
 */
extern SENSOR_RESOLUTION* Sensor_Res[maxMode];
extern CHAR ModulePldInfo[];
extern const SNSR_MIPI_CONFIG DefaultSensorMipiConfig;
extern SNSR_CAPS DefaultSensorCaps;
extern SNSR_RESOLUTION_SET ExportResolutionSet[maxMode];
extern const char* GPIO_STRING[];
extern SENSOR_RESOLUTION sensor_res_still[];
extern SENSOR_RESOLUTION sensor_res_video[];
extern SENSOR_RESOLUTION sensor_res_preview[];
extern SENSOR_RESOLUTION sensor_res_raw[];
extern const LONG ExpEffectDelayFrames[];

#define REGISTRY_INVALID (0xff)

#include "vcm.h"
#include "nvram.h"
#include "SsSensor.h"

/*! "brief InitContext
 *    init context
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
void InitContext(
    PDEVICE_CONTEXT deviceCtx
    );

/*! "brief Detect
 *    detect cmos after power on
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_VERSION_INFO *Version - sensor version information
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Detect(
    PDEVICE_CONTEXT SensorCtx,
    SNSR_VERSION_INFO *Version
    );

/*! "brief ModuleCheck
 *    module detect after power on
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS ModuleCheck(
    PDEVICE_CONTEXT SensorCtx
    );

/*! "brief UnInitCmos
 *    uninit cmos
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS UnInitCmos(
    PDEVICE_CONTEXT pDevExt
    );

/*! "brief InitCmos
 *    init cmos
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS InitCmos(
    PDEVICE_CONTEXT SensorCtx
    );

/*! "brief InitOtp
 *    init otp
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS InitOtp(
    PNVM_CONTEXT pSsNvm
    );

/*! "brief InitRom
 *    init nvram
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS InitRom(
    PNVM_CONTEXT pSsNvm
    );

NTSTATUS
DumpNVM(
    PNVM_CONTEXT pSsNvm,
    ULONG Size
);



/*! "brief ReleaseI2c
 *    release i2c
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
void ReleaseI2c(
    PDEVICE_CONTEXT SensorCtx
    );

/*! "brief ReleaseRom
 *    release nvram 
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
void ReleaseRom(
    PNVM_CONTEXT SensorCtx
    );

/*! "brief Stream
 *    streaming on/off
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    BOOLEAN flag - streaming on/off
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Stream(
    PDEVICE_CONTEXT SensorCtx,
    BOOLEAN flag
    );

/*! "brief GetTimingFactor
 *    get time factor
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PSENSOR_MODE_DATA pModeData - mode data buffer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS GetTimingFactor(
    PDEVICE_CONTEXT SensorCtx,
    PSENSOR_MODE_DATA pModeData
);

/*! "brief InitResolution
 *    init resolution
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_MODE Mode - sensor mode 
 *    INT32 idx - resolution index 
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS InitResolution(
    PDEVICE_CONTEXT SensorCtx,
    SNSR_MODE Mode,
    INT32 idx
    );

/*! "brief SensorPower
 *    sensor power on/off
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    BOOLEAN PowerOn - power on/off
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SensorPower(
    PDEVICE_CONTEXT SensorCtx,
    BOOLEAN PowerOn
    );

/*! "brief SensorPowerAllOff
 *    sensor all power off
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SensorPowerAllOff(
    PDEVICE_CONTEXT SensorCtx
    );

/*! "brief expSetExposure
 *    exposure time setting, for sync up purpose
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_EXPOSURE ExposureTime - exposure time
 *    EXP_STATUS_EN ExpStatus - exposure time status
 *    EXP_STATUS_EN GainStatus - AG status
 *    EXP_STATUS_EN DGainStatus - DG status
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS expSetExposure(
    PDEVICE_CONTEXT SensorCtx,
    SNSR_EXPOSURE ExposureTime,
    EXP_STATUS_EN ExpStatus,
    EXP_STATUS_EN GainStatus,
    EXP_STATUS_EN DGainStatus
    );

/*! "brief expReadYmean
 *    exposure time read y mean
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    ULONG* pYmean - y mean
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS expReadYmean(
    PDEVICE_CONTEXT SensorCtx,
    ULONG* pYmean
    );

/*! "brief aeCmd_SetExposure
 *    exposure time setting, for sync up purpose
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_EXPOSURE ExposureTime - exposure time
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS aeCmd_SetExposure(
    PDEVICE_CONTEXT SensorCtx,
    SNSR_EXPOSURE ExposureTime
    );

/*! "brief UpdateContext
 *    update context of nvram
 *
 * Arg:
 *    PNVM_CONTEXT SensorCtx - context pointer of nvram
 *    UINT16 Addr - address of nvram
 *    UINT8* pData - data buffer pointer of nvram
 *    INT32 Length - data length
 * Return:
 *    NTSTATUS - status
 */
void UpdateContext(
    PNVM_CONTEXT SensorCtx,
    UINT16 Addr,
    UINT8* pData,
    INT32 Length
    );

/*! "brief Crc16
 *    update context of nvram
 *
 * Arg:
 *    unsigned char *Data - data buffer pointer of nvram
 *    unsigned long Len - data length
 * Return:
 *    NTSTATUS - status
 */
unsigned short Crc16(
    unsigned char *Data, 
    unsigned long Len
    );

#ifdef SKY_CAM
/*! "brief QueryIvcamSCLDeviceInterface
 *    query ivcam scl device interface
 *
 * Arg:
 *    PDEVICE_OBJECT deviceObject - device object
 *    PSENSOR_CONTROL_LINK intf - interface
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS QueryIvcamSCLDeviceInterface(
    PDEVICE_OBJECT deviceObject, 
    PSENSOR_CONTROL_LINK intf
    );
    
/*! "brief CleanUpIvcamSCLInterfaceContext
 *    clean up ivcam scl interface context
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS CleanUpIvcamSCLInterfaceContext(
    PDEVICE_CONTEXT deviceCtx
    );

#endif

#ifdef WA_H1504132077
extern ULONG g_FpsForDualCam;

/*! "brief IsDualCamMode
 *    is dual camera mode
 *
 * Return:
 *    BOOLEAN - true/false
 */
BOOLEAN IsDualCamMode(
    );

/*! "brief DuplicateRegsSetting
 *    duplicate register setting
 *
 * Arg:
 *    const REGS_SETTING *regs - sensor setting
 * Return:
 *    REGS_SETTING *  - sensor setting
 */
REGS_SETTING * DuplicateRegsSetting(
    const REGS_SETTING *regs
    );

/*! "brief DestoryRegsSetting
 *    destroy register setting
 *
 * Arg:
 *    const REGS_SETTING *regs - sensor setting
 */
void DestoryRegsSetting(
    REGS_SETTING *regs
    );

/*! "brief SetFps
 *    set fps
 *
 * Arg:
 *    ULONG fps - fps to be set
 *    ULONGLONG pclk - pixel clock
 *    REGS_SETTING * regs - sensor setting
 *    ULONG regHTimingSizeH - reg setting of HTS High
 *    ULONG regHTimingSizeL - reg setting of HTS Low
 *    ULONG regVTimingSizeH - reg setting of VTS High
 *    ULONG regVTimingSizeL - reg setting of VTS Low
 */
void SetFps(
    ULONG fps,
    ULONGLONG pclk,
    REGS_SETTING * regs,
    ULONG regHTimingSizeH,
    ULONG regHTimingSizeL,
    ULONG regVTimingSizeH,
    ULONG regVTimingSizeL
    );
#endif

#endif

