//*****************************************************************************
//                  Copyright (c) 2008, Lexmark International, Inc.           
//     THIS IS AN UNPUBLISHED WORK CONTAINING CONFIDENTIAL AND PROPRIETARY     
//        INFORMATION WHICH IS THE PROPERTY OF LEXMARK INTERNATIONAL, INC.     
//                                                                             
//   ANY DISCLOSURE, USE, OR REPRODUCTION, WITHOUT WRITTEN AUTHORIZATION FROM  
//              LEXMARK INTERNATIONAL, INC., IS STRICTLY PROHIBITED.           
//*****************************************************************************
// scale.js provides font size and image scaling functions.
//
//****************************************************************************

// Current frame dimensions
var actualFrameHeight = null;
var actualFrameWidth = null;

// Design time constants
var designFrameWidth = 440;
var designJSWFrameHeight = 50;
var designBBFrameHeight = 125;

// Scale factor variables
var xScale = 1;
var yScale = 1;

var scaledJSW = false;
var scaledBB = false;

//-----------------------------------------------------------------------------
//  IsScalingRequired()
//   A method that checks the current frame's width against the designed width.
//  Return:
//   bRequired  True - widths are not the same. Scaling is needed.
//              False - widths are equal. Scaling is not needed.
//-----------------------------------------------------------------------------
function IsScalingRequired()
{
	var bRequired = false;

	// Get the current width
	actualFrameWidth = document.body.offsetWidth;

	if (actualFrameWidth != designFrameWidth)
	{
		bRequired = true;
	}

	return bRequired;
}

//-----------------------------------------------------------------------------
// ScaleImages
//   Given a frame in which to work (e.g. window or window.contentframe), this
//   will scale all img's in that frame by altering their width based on the
//   already-set xScale variable.
//  Params:
//   frame        The frame (e.g. window or window.contentframe) in which
//                images should be scaled.
//-----------------------------------------------------------------------------
function ScaleImages(frame)
{
    var images = frame.document.getElementsByTagName('img');
    var image, limit = images.length;
    
    // Scale images within the frame
    for (var i=0; i<limit; i++)
    {
        image = images[i];
        image.width = GetScaledImageWidth(image.width);
    }
}

//-----------------------------------------------------------------------------
// ScaleTables
//   Given a frame in which to work (e.g. window or window.contentframe), this
//   will scale the width of all tables in that frame using the ScaleTable()
//   function.
//  Params:
//   frame        The frame (e.g. window or window.contentframe) in which to
//                scale tables.
//-----------------------------------------------------------------------------
function ScaleTables(frame)
{
    var tables = frame.document.getElementsByTagName('table');
	var table, limit = tables.length;
	
	for (var i=0; i<limit; i++)
	{
	    table = tables[i];
	    ScaleTable(table);
	}
}

//-----------------------------------------------------------------------------
// ScaleTable
//   Given a table to scale, this will scale the height of all its rows based
//   on the already-set yScale variable.
//  Params:
//   table        The table whose rows you wish to scale by height.
//-----------------------------------------------------------------------------
function ScaleTable(table)
{
    var rows = table.getElementsByTagName('tr');
    var curHeight, row;
    var limit = rows.length;
    
    // Scale how far the table is from the left
    var curLeft = parseInt(table.currentStyle.left);
    if (curLeft)
    {
        table.style.left = Math.round(curLeft * xScale) + "px";
    }
    
    // Scale how far the table is from the bottom
    var curBottom = parseInt(table.currentStyle.bottom);
    if (curBottom)
    {
        table.style.bottom = Math.round(curBottom * yScale) + "px";
    }
	
	// Scale the height of each row, where it is defined
    for (var i=0; i<limit; i++)
    {
        row = rows[i];
        curHeight = parseInt(row.currentStyle.height);
        
        if (curHeight)
        {
            row.style.height = Math.round(curHeight * yScale) + "px";
        }
    }
}

//-----------------------------------------------------------------------------
// ScaleJSW
//   Scales the upper part of the window if necessary, including the logo, the
//   cancel button, and the text.
//-----------------------------------------------------------------------------
function ScaleJSW()
{
	var frame = window.JSWFrame;
    var logo = frame.document.getElementById('Logo');
    var jswBody = frame.document.getElementsByTagName('body')[0];
    if (logo)
    {
        if (jswBody)
        {
            if (jswBody.getAttribute("dir") != "rtl")
            {
    	        logo.style.right = "0px";
            }
            else
            {
    	        logo.style.left = "0px";
            }
        }
    }
    
    if (scaledJSW || !IsScalingRequired())
    {
        return;
    }
    
    actualFrameHeight = document.body.offsetHeight;
	xScale = actualFrameWidth / designFrameWidth;
	yScale = actualFrameHeight / designJSWFrameHeight;
	
	ScaleIframes();
	
	ScaleBody(frame);
    ScaleTables(frame);
    ScaleDivs(frame);
    
    var element, curSize;
	var jobStatus = frame.document.getElementById("JobStatus");
	var inkStatus = frame.document.getElementById("InkStatus");
	var textPercent = frame.document.getElementById("TextPercent");
	var filePath = frame.document.getElementById("FilePath");
    var progressPercent = frame.document.getElementById("ProgressPercent");
	var cancelButton = frame.document.getElementById("CancelButton");
    var jswBG = frame.document.getElementById('blackbar');
    var textElements = [jobStatus, inkStatus, textPercent, filePath,
        progressPercent];
    
    // Scale the font size for various text elements
    for (var i=0; i<textElements.length; i++)
    {
        element = textElements[i];
        
	    if (element)
	    {
	        curSize = parseInt(element.currentStyle.fontSize);
		    element.style.fontSize = GetScaledFontSize(curSize);
	    }
	}

    // Scale the cancel button width, and the height will adjust accordingly
	if (cancelButton)
	{
	    cancelButton.width = GetScaledImageWidth(cancelButton.width);
	}
	
    if (logo)
    {
        // Scale the logo
        logo.width = GetScaledImageWidth(logo.width);
    
        if (jswBG)
        {
            // Make the background image's height match that of the scaled logo
            // image so that they line up correctly.
            jswBG.height = logo.offsetHeight;
        }
    }
	
    scaledJSW = true;
}

//-----------------------------------------------------------------------------
// ScaleBB()
//  This method will scale the billboard part of the window if necessary.
//-----------------------------------------------------------------------------
function ScaleBB()
{
	// Determine if we need to scale
	if (scaledBB || !IsScalingRequired() || frames['BillBoardFrame'] == null)
	{
	    return;
	}
	
	var frame = frames['BillBoardFrame'];
	actualFrameHeight = frame.document.body.offsetHeight;
	xScale = actualFrameWidth / designFrameWidth;
	yScale = actualFrameHeight / designBBFrameHeight;
	
	ScaleBody(frame);
	ScaleTables(frame);
	ScaleDivs(frame);
	ScaleImages(frame);
	
	var element, curSize;
	var inkBanner = frame.document.getElementById('InkBanner');
	var orderInk = frame.document.getElementById('OrderInk');
	var partNum1 = frame.document.getElementById('partNum1');
	var partNum2 = frame.document.getElementById('partNum2');
	var partNum3 = frame.document.getElementById('partNum3');
	var partNum4 = frame.document.getElementById('partNum4');
	var dykText = frame.document.getElementById('dyktextcell');
	var cartridgeHeader = frame.document.getElementById('cartridgeheader');
	var qualityHeader = frame.document.getElementById('qualityheader');
	var photoHeader = frame.document.getElementById('photoheader');
	var textElements = [inkBanner, orderInk, partNum1, partNum2,
	    partNum3, partNum4, dykText, cartridgeHeader, qualityHeader,
	    photoHeader];
	
	for (var i=0; i<textElements.length; i++)
	{
	    element = textElements[i];
	    
	    if (element)
	    {
	        // font size
	        curSize = parseInt(element.currentStyle.fontSize);
	        if (curSize)
	        {
		        element.style.fontSize = GetScaledFontSize(curSize);
		    }
		    
		    // width
		    curSize = parseInt(element.currentStyle.width);
		    if (curSize)
		    {
		        element.style.width = Math.round(curSize * xScale) + "px";
		    }
	    }
	}
	
	var message = frame.document.getElementById('Message');
	if (message)
	{
	    // Adjust the width of the warning message
	    curSize = parseInt(message.currentStyle.width);
        if (curSize)
        {
            message.style.width = Math.round(curSize * xScale) + "px";
        }
        
        // Adjust the position from the right for the warning message
        curSize = parseInt(message.currentStyle.right);
        if (curSize)
        {
            message.style.right = Math.round(curSize * xScale) + "px";
        }
        
        // Adjust the position from the bottom for the warning message
        curSize = parseInt(message.currentStyle.bottom);
        if (curSize)
        {
            message.style.bottom = Math.round(curSize * yScale) + "px";
        }
    }
    
    if (inkBanner)
    {
        // Adjust the height of the containing div that holds the Order Ink
	    // button and the warning message.
	    curSize = parseInt(inkBanner.currentStyle.height);
        if (curSize)
        {
            inkBanner.style.height = Math.round(curSize * yScale) + "px";
        }
    }
    
    var inkContainer = frame.document.getElementById('InkContainer');
    if (inkContainer)
    {
        // Scale the padding between each ink container image
        var cells = inkContainer.getElementsByTagName('td'), cell;
        for (var i=0; i<cells.length; i++)
        {
            cell = cells[i];
            if (cell)
            {
                curSize = parseInt(cell.currentStyle.paddingRight);
                if (curSize)
                {
                    cell.style.paddingRight = Math.round(curSize * xScale) + "px";
                }
            }
        }
    }
    
    if (orderInk)
    {
        // Adjust the position from the right for the Order Ink button
        curSize = parseInt(orderInk.currentStyle.right);
        if (curSize)
        {
            orderInk.style.right = Math.round(curSize * xScale) + "px";
        }
    	
        // Adjust the position from the bottom for the Order Ink button
        curSize = parseInt(orderInk.currentStyle.bottom);
        if (curSize)
        {
            orderInk.style.bottom = Math.round(curSize * xScale) + "px";
        }
    
        // Adjust the height of the Order Ink button
        var orderInkHeight = parseInt(orderInk.currentStyle.height);
        if (curSize)
        {
            orderInk.style.height = Math.round(orderInkHeight * yScale) + "px";
        }
    }
    
	scaledBB = true;
}

//-----------------------------------------------------------------------------
// ScalePageCopy
//   Scales the font size of the page count and copy count if those elements
//   exist and need scaling.
//  Params:
//   elementPageCount        Some HTML element containing the page count for
//                           this print job.
//   elementCopyCount        Some HTML element containing the copy count for
//                           this print job.
//-----------------------------------------------------------------------------
function ScalePageCopy(elementPageCount, elementCopyCount)
{
    if (!IsScalingRequired())
	{
	    return;
	}
	
	var frame = window.JSWFrame;
	
    if (null != elementPageCount)
    {
        // Get the scaled font size for the PageCount text
        scaledSize = GetScaledFontSize(
            parseInt(elementPageCount.currentStyle.fontSize)
        );
        elementPageCount.style.fontSize = scaledSize;
    }

    if (null != elementCopyCount)
    {
        // Get the scaled font size for the CopyCount text
        scaledSize = GetScaledFontSize(
            parseInt(elementCopyCount.currentStyle.fontSize)
        );
        elementCopyCount.style.fontSize = scaledSize;
    }
}

//-----------------------------------------------------------------------------
// ScaleBody
//   Finds the first body element within the given frame (really, there should
//   only be one 'body' element in an HTML page) and scales the padding on it.
//  Params:
//   frame        The frame in which a body element should be searched for,
//                e.g. window or frames['my_frame'].
//-----------------------------------------------------------------------------
function ScaleBody(frame)
{
    // Get all body elements in the given frame
    var bodies = frame.document.getElementsByTagName("body");
    
    // Ensure there's at least one body to work with
    if (bodies.length < 1)
    {
        return;
    }
    
    // Grab the first body, which is the only one we're going to work with
    var body = bodies[0];
    
    // Get the current padding on the body
	var curLeft = parseInt(body.currentStyle.paddingLeft);
	var curRight = parseInt(body.currentStyle.paddingRight);
	var curTop = parseInt(body.currentStyle.paddingTop);
	var curBottom = parseInt(body.currentStyle.paddingBottom);
	
	// Scale the padding on the left side of the body
	if (curLeft)
	{
	    body.style.paddingLeft = Math.round(curLeft * xScale) + "px";
	}
	
	// Scale the padding on the right side of the body
	if (curRight)
	{
	    body.style.paddingRight = Math.round(curRight * xScale) + "px";
	}
	
	// Scale the padding on the top side of the body
	if (curTop)
	{
	    body.style.paddingTop = Math.round(curTop * yScale) + "px";
	}
	
	// Scale the padding on the bottom side of the body
	if (curBottom)
	{
	    body.style.paddingBottom = Math.round(curBottom * yScale) + "px";
	}
}

//-----------------------------------------------------------------------------
// ScaleIframes
//   Scales the height on the JSW and the billboard iframes.
//-----------------------------------------------------------------------------
function ScaleIframes()
{
    var iframe, curHeight, newHeight;

    // An array of ID's that represent iframes in the page
	var iframes = ['JSWFrame', 'BillBoardFrame'];
	
    // Go through each of the iframe ID's
    for (var i=0; i<iframes.length; i++)
    {
        // Get the iframe associated with the current ID
        iframe = document.getElementById(iframes[i]);
        
        // Get an integer representation of the current height that is set on
        // this iframe
        curHeight = parseInt(iframe.currentStyle.height);
        
        // If a height is set...
        if (curHeight)
        {
            // Calculate a new height by scaling the old height using the
            // y-axis scale factor
            newHeight = Math.round(curHeight * yScale);
            
            // If the two heights are different...
            if (curHeight != newHeight)
            {
                // Set the iframe's height to the new height in pixels
                iframe.style.height = newHeight + "px";
            }
        }
	}
}

//-----------------------------------------------------------------------------
// ScaleDivs
//   Given a frame in which to work (e.g. window or window.contentframe), this
//   will scale the width of all div layers in that frame, based on the
//   already-initialized xScale variable.  Scaling the height as well as the
//   width was not necessary and actually messed up the appearance of the page.
//  Params:
//   frame        The frame (e.g. window or window.contentframe) in which to
//                scale div elements.
//-----------------------------------------------------------------------------
function ScaleDivs(frame)
{
    var divs = frame.document.getElementsByTagName('div');
    var div, limit = divs.length, curVal;
    
    for (var i=0; i<limit; i++)
	{
	    div = divs[i];
	    
	    // width
	    curVal = parseInt(div.currentStyle.width);
	    if (curVal)
        {
            div.style.width = Math.round(curVal * xScale) + "px";
        }
        
        // distance from left
        curVal = parseInt(div.currentStyle.left);
	    if (curVal)
        {
            div.style.left = Math.round(curVal * xScale) + "px";
        }
        
        // distance from top
        curVal = parseInt(div.currentStyle.top);
	    if (curVal)
        {
            div.style.top = Math.round(curVal * yScale) + "px";
        }
        
        // distance from bottom
        curVal = parseInt(div.currentStyle.bottom);
	    if (curVal)
        {
            div.style.bottom = Math.round(curVal * yScale) + "px";
        }
	}
}

//-----------------------------------------------------------------------------
//  GetScaledFontSize  
//   A method for retrieving the scaled font size.
//  Params:
//   origFontSize    The originally designed font size
//  Return:
//   scaledFontSize  The scaled size of the font.
//-----------------------------------------------------------------------------
function GetScaledFontSize(origFontSize)
{
    var scaledFontSize = parseInt(
        origFontSize * xScale
    ) + "px";

	return scaledFontSize;
}

//-----------------------------------------------------------------------------
//  GetScaledImageWidth  
//   A method for retrieving the scaled width of an image.
//  Params:
//   imgWidth        The width of the current image
//  Return:
//   imgScaledWidth  The scaled width of the image.
//-----------------------------------------------------------------------------
function GetScaledImageWidth(imgWidth)
{
    return Math.round(
	    imgWidth * (actualFrameWidth / designFrameWidth)
	);
}
