//
// Copyright (c) 2016 Ricoh Co., Ltd. All rights reserved.
//
// Abstract:
//    RICOH V4 Printer Driver
//

var psfPrefix = "psf";
var pskPrefix = "psk";
var userpskPrefix = "ns0000";

var pskNs = "http://schemas.microsoft.com/windows/2003/08/printing/printschemakeywords";
var psfNs = "http://schemas.microsoft.com/windows/2003/08/printing/printschemaframework";
var ricohNs = "http://schemas.ricoh.com/2007/printing/keywords";

var NsMap = {
    "psk":"http://schemas.microsoft.com/windows/2003/08/printing/printschemakeywords",
    "ns0000":"http://schemas.ricoh.com/2007/printing/keywords"
};

var FRAMEWORK_URI = "http://schemas.microsoft.com/windows/2003/08/printing/printschemaframework";
var KEYWORDS_URI = "http://schemas.microsoft.com/windows/2003/08/printing/printschemakeywords";

var SCHEMA_INST_URI = "http://www.w3.org/2001/XMLSchema-instance";
var SCHEMA_DEF_URI = "http://www.w3.org/2001/XMLSchema";


var SCHEMA_XSI = "xsi";
var SCHEMA_XS = "xsd";

var SCHEMA_TYPE = "type";
var SCHEMA_INTEGER = "integer";
var SCHEMA_DECIMAL = "decimal";
var SCHEMA_STRING = "string";
var SCHEMA_QNAME = "QName";
var SCHEMA_CONDITIONAL = "Conditional";

var FEATURE_ELEMENT_NAME = "Feature";
var OPTION_ELEMENT_NAME = "Option";
var PARAM_INIT_ELEMENT_NAME = "ParameterInit";
var PARAM_REF_ELEMENT_NAME = "ParameterRef";
var PARAM_DEF_ELEMENT_NAME = "ParameterDef";
var SCORED_PROP_ELEMENT_NAME = "ScoredProperty";
var PROPERTY_ELEMENT_NAME = "Property";
var VALUE_ELEMENT_NAME = "Value";
var NAME_ATTRIBUTE_NAME = "name";

var PICKONE_VALUE_NAME = "PickOne";
var SELECTIONTYPE_VALUE_NAME = "SelectionType";
var DATATYPE_VALUE_NAME = "DataType";
var DEFAULTVAL_VALUE_NAME = "DefaultValue";
var MAX_VALUE_NAME = "MaxValue";
var MIN_VALUE_NAME = "MinValue";
var MAX_LENGTH_NAME = "MaxLength";
var MIN_LENGTH_NAME = "MinLength";
var MULTIPLE_VALUE_NAME = "Multiple";
var MANDATORY_VALUE_NAME = "Mandatory";
var UNITTYPE_VALUE_NAME = "UnitType";
var DISPLAYNAME_VALUE_NAME = "DisplayName";

var PRINTTICKET_NAME = "PrintTicket";
var PRINTCAPABILITIES_NAME = "PrintCapabilities";

var PROPERTY_KEY_ENABLE_CHANGEDEFAULT = "JS_Customize_Enabled";


var EVENT_DEVMODETOPRINTTICKET = "1";
var EVENT_PRINTTICKETTODEVMODE = "2";
var EVENT_PRINTCAPABILITIES    = "3";

var global = {};
global.eventtype = null;
global.language  = "unknown";
global.isChangeDefault = null;

var AUTO_CREATE_FEATURE_LIST = [
    "psk:PageOrientation",
    "psk:PageMediaSize",
    "psk:PageOutputColor",
    "psk:JobNUpAllDocumentsContiguously",
    "psk:JobStapleAllDocuments",
    "psk:JobHolePunch",
    "psk:PageMediaType",
    "psk:JobDuplexAllDocumentsContiguously",
    "psk:PageInputBin",
    "psk:DocumentCollate",
    "psk:JobPageOrder",
    "psk:JobOutputBin",
    "ns0000:JobBooklet",
    "ns0000:JobGraphicsMode",
    "ns0000:JobPrintableArea",
    "ns0000:JobImageDirection",
    "ns0000:JobPrintDensity"
];

var AUTO_CREATE_PARAMETERDEF_LIST = [
    "psk:JobCopiesAllDocuments"
];

var AUTO_CREATE_SCOREDPROPERTY_LIST = [
    "psk:PagesPerSheet"
];

var AUTO_CREATE_RESOLUTON_LIST = [
    "psk:PageResolution"
];

var CHANGEABLE_PICKONE_FEATURE_LIST = [
    "ns0000:WatermarkItem"
];

var PSKJOBDUPLEX_CONTIGUOUSLY_MAP = {
    'ns0000:Off': 'psk:OneSided',
    'ns0000:LongEdgeLeftOrTop': 'psk:TwoSidedLongEdge',
    'ns0000:LongEdgeRightOrTop': 'psk:TwoSidedLongEdge',
    'ns0000:ShortEdgeTopOrLeft': 'psk:TwoSidedShortEdge',
    'ns0000:ShortEdgeTopOrRight': 'psk:TwoSidedShortEdge'
};

var RPSKJOBDUPLEX_DIRECTION_MAP = {
    'psk:OneSided':
        {
            'ns0000:Off': 'ns0000:Off' ,
            'ns0000:LongEdgeLeftOrTop': 'ns0000:Off' ,
            'ns0000:LongEdgeRightOrTop': 'ns0000:Off' ,
            'ns0000:ShortEdgeTopOrLeft': 'ns0000:Off' ,
            'ns0000:ShortEdgeTopOrRight': 'ns0000:Off'
        },
    'psk:TwoSidedLongEdge':
        {
            'ns0000:Off': 'ns0000:LongEdgeLeftOrTop' ,
            'ns0000:LongEdgeLeftOrTop': 'ns0000:LongEdgeLeftOrTop' ,
            'ns0000:LongEdgeRightOrTop': 'ns0000:LongEdgeRightOrTop' ,
            'ns0000:ShortEdgeTopOrLeft': 'ns0000:LongEdgeLeftOrTop' ,
            'ns0000:ShortEdgeTopOrRight': 'ns0000:LongEdgeRightOrTop'
        },
    'psk:TwoSidedShortEdge':
        {
            'ns0000:Off': 'ns0000:ShortEdgeTopOrLeft',
            'ns0000:LongEdgeLeftOrTop': 'ns0000:ShortEdgeTopOrLeft',
            'ns0000:LongEdgeRightOrTop': 'ns0000:ShortEdgeTopOrRight',
            'ns0000:ShortEdgeTopOrLeft': 'ns0000:ShortEdgeTopOrLeft',
            'ns0000:ShortEdgeTopOrRight': 'ns0000:ShortEdgeTopOrRight'
        }
}

var RESOLUTION_MAP = {
    'ns0000:_300dpi':
        {
            'ResolutionX': '300',
            'ResolutionY': '300'
        },
    'ns0000:_300to600dpi':
        {
            'ResolutionX': '300',
            'ResolutionY': '300'
        },
    'ns0000:_600dpi':
        {
            'ResolutionX': '600',
            'ResolutionY': '600'
        },
    "ns0000:_1200x600":
        {
            'ResolutionX': '600',
            'ResolutionY': '600'
        },
    "ns0000:_2400x600":
        {
            'ResolutionX': '600',
            'ResolutionY': '600'
        },
    'ns0000:_1200dpi':
        {
            'ResolutionX': '1200',
            'ResolutionY': '1200'
        }
}

var KEY_TAG = "tag";
var KEY_ATTRIBUTE ="attribute";
var KEY_CHILDREN = "children";
var KEY_CHILDFEATURE = "childfeature";
var KEY_XPATH = "xpath";
var KEY_VALUE = "value";
var KEY_DEFAULT = "default";
var KEY_NAME = "name";
var KEY_TYPE = "type";
var KEY_PICKONE = "pickone";
var KEY_BLANK = "blank";
var KEY_MIN = "min";
var KEY_MAX = "max";

var TAG_FEATURE = "psf:Feature";
var TAG_PARAMETERDEF = "psf:ParameterDef";
var TAG_FUNCTION = "function";
var TYPE_STRING = "string";
var TYPE_INTEGER = "integer";

var PTPSD_KEY_PTNAME    = "pt_name";
var PTPSD_KEY_PSDNAME   = "psd_name";
var PTPSD_KEY_ATTRIBUTE = "attribute";
var PTPSD_KEY_PICKONE   = "pickone";

var RCF_ITEM_KEY_NAME     = "name";
var RCF_ITEM_KEY_DEFAULT  = "default";
var RCF_ITEM_KEY_FIXVALUE = "fixvalue";
var RCF_ITEM_KEY_PICKONE  = "pickone";
var RCF_ITEM_KEY_MAX      = "max";
var RCF_ITEM_KEY_VALUE    = "value";
var RCF_ITEM_KEY_ITEMLIST = "items";

/////////////////////////////////////////////
//
//
// Public I/F
//
/////////////////////////////////////////////

////
//// @param printTicket      Print ticket to be validated
//// @param scriptContext    Script context object
////
//// @return Integer value indicating validation status
//// @retval 1 Valid print ticket
//// @retval 2 Print ticket was modified to make it valid
//// @retval 0 Invalid print ticket (not returned by this example)
function validatePrintTicket(printTicket, scriptContext) {
    debugger;

    return 1;
}

////
//// @param printTicket      Print ticket which may constrain the capabilities
//// @param scriptContext    Script context object
//// @param printCapablities Print capabilities object to be expanded
////
//// @return None.
////

//// </summary>
//// <param name="printTicket" type="IPrintSchemaTicket" mayBeNull="true">
////     If not 'null', the print ticket's settings are used to customize the print capabilities.
//// </param>
//// <param name="scriptContext" type="IPrinterScriptContext">
////     Script context object.
//// </param>
//// <param name="printCapabilities" type="IPrintSchemaCapabilities">
////     Print capabilities object to be customized.
//// </param>
function completePrintCapabilities(printTicket, scriptContext, printCapabilities) {

    debugger;
    if (!printCapabilities) {
        return;
    }

    try {
        global.eventtype = EVENT_PRINTCAPABILITIES;

        setCurrentLanguage( printCapabilities );

        CreateAllPrintCapabilities(printTicket, scriptContext, printCapabilities);

    }
    catch (e) {

        var result = 0;
        result = e.number;
        Debug.write(e.message + "completePrintCapabilities() Error!");        
    }

    return;
}

////
//// @param printTicket      Print ticket to be converted to DevMode
//// @param scriptContext    Script context object
//// @param devMode          The DevMode property bag (Persistent storage for properties/data)
////
//// @return None.
////
function convertPrintTicketToDevMode(printTicket, scriptContext, devModeProperties) {

    debugger;
    try
    {
        convertAllFeaturesPrintTicketToDevmode( printTicket, scriptContext, devModeProperties);
        
        validatePrintTicketToDevmode( printTicket, scriptContext );
    }
    catch (e) {

        var result = 0;
        result = e.number;
        Debug.write(e.message + "convertPrintTicketToDevMode() Error!");        
    }
}

////
//// @param devMode          The DevMode property bag (Persistent storage for properties/data)
//// @param scriptContext    Script context object
//// @param printTicket      The output print ticket
////
//// @return None.
////
function convertDevModeToPrintTicket(devModeProperties, scriptContext, printTicket) {    

    debugger;
    try {
        global.eventtype = EVENT_DEVMODETOPRINTTICKET;
        global.isChangeDefault = GetBoolFromProperties(scriptContext.QueueProperties, PROPERTY_KEY_ENABLE_CHANGEDEFAULT );

        createAllPrintTicket(devModeProperties, scriptContext, printTicket);

        validateDevmodeToPrintTicket(printTicket, scriptContext);
    }
    catch (e) {
        var result = 0;
        result = e.number;
        Debug.write(e.message + "convertDevModeToPrintTicket() Error!");
    }
}



/////////////////////////////////////////////
//
//
// Private I/F
//
/////////////////////////////////////////////

function getPrintTicketTreeTable( pcTable, property, scriptContext )
{
    var tree = [];
    try
    {
        for(var i=0; i<pcTable.length; i++)
        {
            var element = null;
            if( TAG_FEATURE == pcTable[i][KEY_TAG] )
            {
                element = createFeatureElement( pcTable[i], property );
                if( element !== null )
                {
                    for( var j=0; j<element.length; j++)
                    {
                        tree.push(element[j]);
                    }
                }
            }
            else if( TAG_PARAMETERDEF === pcTable[i][KEY_TAG] )
            {
                element = createParameterDefElement( pcTable[i], property );
                if( element !== null )
                {
                    tree.push(element);
                }
            }
            else if( TAG_FUNCTION === pcTable[i][KEY_TAG] )
            {
                var func = (new Function('arg1', 'arg2', "return " + pcTable[i][KEY_ATTRIBUTE] + "(arg1, arg2)"));
                element = func( scriptContext, property );
                if( element !== null )
                {
                    tree.push(element);
                }
            }
        }
    }
    catch(e) {
        tree = null;
    }

    return tree;
}

function createFeatureElement( pcTable, property )
{
    var featureElements = [];
    try
    {
        var feature = null;
        var options = null;
        var childFeature = null;

        if( Object.prototype.toString.call(pcTable) === '[object Object]' )
        {
            if( TAG_FEATURE == pcTable[KEY_TAG] &&
                (  property.isSupported( pcTable[KEY_ATTRIBUTE] ) ||
                   ( KEY_PICKONE in pcTable && "blank" in pcTable[KEY_PICKONE] ) ||
                   KEY_CHILDFEATURE in pcTable ) )
            {
                feature = new FeatureElement( property, pcTable[KEY_ATTRIBUTE] );

                options = createOptionElement( pcTable[KEY_PICKONE], property, feature.getName() );
                if( options !== null )
                {
                    for(var i=0; i<options.length; i++ )
                    {
                        feature.appendChild( options[i] );
                    }
                }

                childFeature  = createFeatureElement( pcTable[KEY_CHILDFEATURE], property );
                if( childFeature )
                {
                    for(var j=0; j<childFeature.length; j++)
                    {
                        feature.appendFeature( childFeature[j] );
                    }
                }
                if( (options !== null && options.length > 0) ||
                    (childFeature !== null && childFeature.length > 0 ) )
                {
                    featureElements.push( feature );
                }
            }
        }
        else if( Object.prototype.toString.call(pcTable) === '[object Array]' )
        {
            for(var i=0; i<pcTable.length; i++)
            {
                if( TAG_FEATURE === pcTable[i][KEY_TAG] &&
                    ( property.isSupported( pcTable[i][KEY_ATTRIBUTE] ) || ( KEY_PICKONE in pcTable[i] && KEY_BLANK in pcTable[i][KEY_PICKONE] ) ) )
                {
                    feature = new FeatureElement( property, pcTable[i][KEY_ATTRIBUTE] );

                    options = createOptionElement( pcTable[i][KEY_PICKONE], property, feature.getName() );
                    if( options !== null )
                    {
                        for(var j=0; j<options.length; j++ )
                        {
                            feature.appendChild( options[j] );
                        }
                    }

                    childFeature  = createFeatureElement( pcTable[i][KEY_CHILDFEATURE], property );
                    if( childFeature && feature )
                    {
                        for(var j=0; j<childFeature.length; j++)
                        {
                            feature.appendFeature( childFeature[j] );
                        }
                    }

                    if( (options !== null && options.length > 0) ||
                        (childFeature !== null && childFeature.length > 0 ) )
                    {
                        featureElements.push( feature );
                    }
                }
            }
        }
    }
    catch(e) {
        featureElements = null;
    }
        
    return featureElements;
}


function createOptionElement( optionTable, property, featureName )
{
    var options = [];
    try
    {
        var pickoneList = property.getPickoneList(featureName);

        var option = null;
        var scoredElements = null;
        var optionPickone = null;
        
        if( pickoneList === null )
        {
            if( KEY_BLANK in optionTable )
            {
                option = new OptionElement( "", property );

                if( option !== null )
                {
                    scoredElements = createScoredPropertyElement( optionTable[KEY_BLANK], property );
                    if( scoredElements.length > 0 )
                    {
                        if( scoredElements.length >= 2 &&
                            scoredElements[0].getName() == scoredElements[1].getName() )
                        {
                            for(var j=0; j<scoredElements.length; j++) {
                                optionPickone = new OptionElement( "", property );
                                optionPickone.appendChild( scoredElements[j] );
                                options.push( optionPickone );
                            }
                        }
                        else
                        {
                            for(var j=0; j<scoredElements.length; j++) {
                                option.appendChild( scoredElements[j] );
                            }
                            options.push( option );
                        }
                    }
                }
            }
        }
        else
        {
            for (var i = 0; i < pickoneList.length; i++)
            {
                var name = pickoneList[i];

                option = null;
                
                if( containsInArray( name, pickoneList ) )
                {
                    option = new OptionElement( name, property );
                    if( option ) {
                        options.push( option );
                    }
                }

                if ((option !== null) && (optionTable !== undefined))
                {
                    scoredElements = createScoredPropertyElement(optionTable[name], property);

                    if( scoredElements.length >= 2 &&
                        scoredElements[0].getName() == scoredElements[1].getName() )
                    {
                        options.pop();
                        for(var j=0; j<scoredElements.length; j++) {
                            optionPickone = new OptionElement( name, property );
                            optionPickone.appendChild( scoredElements[j] );
                            options.push( optionPickone );
                        }
                    }
                    else
                    {
                        for (var j = 0; j < scoredElements.length; j++)
                        {
                            option.appendChild(scoredElements[j]);
                        }
                    }
                }
            }
        }
    }
    catch(e) {
        options = [];
    }
        
    return options;
}

function isBlankObject( obj )
{
    if(obj === undefined)
    {
        return false;
    }
    else if(obj.blank === undefined)
    {
        return false;
    }
    else
    {
        return true;
    }
}

function isArrayObject( obj )
{
    if( Object.prototype.toString.call(obj) === '[object Array]' )
    {
        return true;
    }
    else
    {
        return false;
    }
}


function createScoredPropertyElement( scoredTable, property )
{
    var scoredElements = [];
    try
    {
        var isPickOneProperty = false;
        if (scoredTable.length > 2 && scoredTable[0][KEY_NAME] == scoredTable[1][KEY_NAME])
        {
            isPickOneProperty = true;
        }
        for(var i=0; i<scoredTable.length; i++ )
        {
            var scored = null;
            if( "ref"  === scoredTable[i][KEY_TYPE] )
            {
                if( property.isSupported( scoredTable[i][KEY_VALUE] ) )
                {
                    scored = new ScoredPropertyParameterRefElement(
                            scoredTable[i][KEY_NAME],
                            scoredTable[i][KEY_VALUE],
                            property
                            );
                }
            }
            else if( "integer" === scoredTable[i][KEY_TYPE] )
            {
                if (isPickOneProperty )
                {
                    if( property.isSupported(scoredTable[i][KEY_NAME]) )
                    {
                        scored = new ScoredPropertyIntegerElement(
                                scoredTable[i][KEY_NAME],
                                scoredTable[i][KEY_VALUE],
                                property
                                );
                    }
                }
                else
                {
                    scored = new ScoredPropertyIntegerElement(
                            scoredTable[i][KEY_NAME],
                            scoredTable[i][KEY_VALUE],
                            property
                            );
                }
            }
            else if( "string" == scoredTable[i][KEY_TYPE] )
            {
                scored = new ScoredPropertyStringElement(
                        scoredTable[i][KEY_NAME],
                        scoredTable[i][KEY_VALUE],
                        property
                        );
            }

            if( scored ) {
                scoredElements.push( scored );
            }
        }
    }
    catch(e) {
        scoredElements = [];
    }

    return scoredElements;
}

function createParameterDefElement( pcTable, property )
{
    var parameterDef = null;
    try
    {
        if( TAG_PARAMETERDEF == pcTable[KEY_TAG] && property.isSupported( pcTable[KEY_ATTRIBUTE] )  )
        {
            var type;
            var name = pcTable[KEY_ATTRIBUTE];
            var value, unittype, mandatory, min, max, defaultvalue, display, multiple;

            type      = pcTable[KEY_CHILDREN][DATATYPE_VALUE_NAME];
            unittype  = pcTable[KEY_CHILDREN][UNITTYPE_VALUE_NAME];
            mandatory = pcTable[KEY_CHILDREN][MANDATORY_VALUE_NAME];

            if( MIN_VALUE_NAME in pcTable[KEY_CHILDREN] ) {
                min       = pcTable[KEY_CHILDREN][MIN_VALUE_NAME];
            }
            else if( MIN_LENGTH_NAME in pcTable[KEY_CHILDREN] ) {
                min       = pcTable[KEY_CHILDREN][MIN_LENGTH_NAME];
            }

            max = property.getMaxValue(name);
            if( !max )
            {
                if( MAX_VALUE_NAME in pcTable[KEY_CHILDREN] ) {
                    max       = pcTable[KEY_CHILDREN][MAX_VALUE_NAME];
                }
                else if( MAX_LENGTH_NAME in pcTable[KEY_CHILDREN] ) {
                    max       = pcTable[KEY_CHILDREN][MAX_LENGTH_NAME];
                }
            }

            if( MULTIPLE_VALUE_NAME in pcTable[KEY_CHILDREN] ) {
                multiple  = pcTable[KEY_CHILDREN][MULTIPLE_VALUE_NAME];
            }

            if( DISPLAYNAME_VALUE_NAME in pcTable[KEY_CHILDREN] ) {
                display   = pcTable[KEY_CHILDREN][DISPLAYNAME_VALUE_NAME];
            }

            defaultvalue = property.getDefault( name );
            value = property.getCurrentValue( name );

            if( type == "xsd:integer" ) {
                parameterDef = new ParameterDefIntegerElement(
                                    property,
                                    name,
                                    value,
                                    unittype,
                                    mandatory,
                                    min,
                                    max,
                                    defaultvalue,
                                    multiple,
                                    display
                                    );
            }
            else if( type == "xsd:string" ) {
                parameterDef = new ParameterDefStringElement(
                                    property,
                                    name,
                                    value,
                                    unittype,
                                    mandatory,
                                    min,
                                    max,
                                    defaultvalue,
                                    display
                                    );
            }
        }
    }
    catch(e)
    {
        parameterDef = null;
    }
    return parameterDef;
}


var FeatureElement = function( property, name, display )
{
    this.property = property;
    this.name = name;
    this.display = display;
    this.children = [];
    this.childFeature = [];
};

FeatureElement.prototype.getName = function() {
    return this.name;
};

FeatureElement.prototype.appendChild = function( element )
{
    this.children.push( element );
};

FeatureElement.prototype.appendFeature = function( element )
{
    this.childFeature.push( element );
};

FeatureElement.prototype.createPrintTicket = function(rootNode, parentNode){
    //debugger;
    var ret = false;
    try
    {
        var node = null;
        if( !(containsInArray( this.name, AUTO_CREATE_FEATURE_LIST ) ) )
        {
            if( !(containsInArray( this.name, AUTO_CREATE_RESOLUTON_LIST ) ) )
            {
                removeChildNode( rootNode, "psf:PrintTicket/psf:Feature[@name='" + this.name + "']" );

                node = CreateFeature(
                                rootNode,
                                this.name.substring( this.name.indexOf(":") + 1),
                                null,  // displayName
                                this.name.substring( 0, this.name.indexOf(":") )
                                );

                if( node )
                {
                    var value = this.property.getCurrentValue( this.name );
                    for(var i=0; i < this.children.length; i++ )
                    {
                        if( value == this.children[i].getName() ||
                            value === null )
                        {
                            ret = this.children[i].createPrintTicket(rootNode, node );
                            if( ret ) {
                                break;
                            }
                        }
                    }
                }
            }
        }
        else
        {
            node = rootNode.XmlNode.selectSingleNode("psf:PrintTicket/psf:Feature[@name='" + this.name + "']");
        }

        for(var i=0; i < this.childFeature.length; i++ )
        {
            ret = this.childFeature[i].createPrintTicket(rootNode, node );
        }

        if( ret )
        {
            parentNode.appendChild( node );
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

FeatureElement.prototype.createPrintCapabilities = function(rootNode, parentNode){
    var ret = false;
    try
    {
        var node = null;
        if( !(containsInArray( this.name, AUTO_CREATE_FEATURE_LIST ) ) )
        {
            removeChildNode( rootNode, "psf:PrintCapabilities/psf:Feature[@name='" + this.name + "']" );

            node = CreateFeatureSelection(
                            rootNode,
                            this.name.substring( this.name.indexOf(":") + 1),
                            null,
                            this.name.substring( 0, this.name.indexOf(":") )
                            );

            if( node )
            {
                for(var i=0; i < this.children.length; i++ )
                {
                    ret = this.children[i].createPrintCapabilities(rootNode, node );
                }
            }
        }
        else
        {
            node = rootNode.XmlNode.selectSingleNode("psf:PrintCapabilities/psf:Feature[@name='" + this.name + "']");
        }

        if( node )
        {
            for(var i=0; i < this.childFeature.length; i++ )
            {
                ret = this.childFeature[i].createPrintCapabilities(rootNode, node );
            }
        }

        if( ret )
        {
            parentNode.appendChild( node );
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

var OptionElement = function( name, property, display )
{
    this.name = name;
    this.property = property;
    this.display = display;
    this.children = [];
};

OptionElement.prototype.getName = function()
{
    return this.name;
};

OptionElement.prototype.appendChild = function( element )
{
    this.children.push( element );
};

OptionElement.prototype.createPrintTicket = function(rootNode, parentNode ){
    var ret = false;
    try
    {
        var node = CreateOption(
                        rootNode,
                        this.name.substring( this.name.indexOf(":") + 1),
                        null, 
                        this.name.substring( 0, this.name.indexOf(":") )
                        );

        if( node )
        {
            ret = true;
            if( this.children.length > 0 &&
                this.property.getPickoneList( this.children[0].getName() ) !== null )
            {
                for(var i=0; i < this.children.length; i++ )
                {
                    ret = this.children[i].createPrintTicket(rootNode, node );
                    if( ret )
                    {
                        break;
                    }
                }

            }
            else
            {
                for(var i=0; i < this.children.length; i++ )
                {
                    ret = this.children[i].createPrintTicket(rootNode, node );
                }
            }
        }

        if( ret )
        {
            parentNode.appendChild( node );
        }

    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

OptionElement.prototype.createPrintCapabilities = function(rootNode, parentNode ){
    var ret = false;
    try
    {
        var node = CreateOption(
                        rootNode,
                        this.name.substring( this.name.indexOf(":") + 1),
                        null, 
                        this.name.substring( 0, this.name.indexOf(":") )
                        );

        if( node )
        {
            ret = true;
            for(var i=0; i < this.children.length; i++ )
            {
                ret = this.children[i].createPrintCapabilities(rootNode, node );
            }
        }

        if( ret )
        {
            parentNode.appendChild( node );
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

var ScoredPropertyIntegerElement = function( name, value, property)
{
    this.name = name;
    this.value = value;
    this.property = property;
};

var ScoredPropertyStringElement = function( name, value, property)
{
    this.name = name;
    this.value = value;
    this.property = property;
};

var ScoredPropertyParameterRefElement = function( name, refName, property)
{
    this.name = name;
    this.ref = refName;
    this.property = property;
};

ScoredPropertyParameterRefElement.prototype.createPrintTicket = function(rootNode, parentNode){
    var ret = false;
    try
    {
        var node = CreateScoredPropertyRefElement(
                        rootNode,
                        this.name.substring( this.name.indexOf(":") + 1),
                        this.ref.substring( this.ref.indexOf(":") + 1),
                        this.name.substring( 0, this.name.indexOf(":") )
                        );

        if( node )
        {
            this.property.setReferedParameterDef( this.ref );
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};


ScoredPropertyParameterRefElement.prototype.createPrintCapabilities = function(rootNode, parentNode){
    var ret = false;
    try
    {
        var node = CreateScoredPropertyRefElement(
                        rootNode,
                        this.name.substring( this.name.indexOf(":") + 1),
                        this.ref.substring( this.ref.indexOf(":") + 1),
                        this.name.substring( 0, this.name.indexOf(":") )
                        );

        if( node )
        {
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};


ScoredPropertyParameterRefElement.prototype.getRefName = function() {
    return this.ref;
};

ScoredPropertyParameterRefElement.prototype.getName = function() {
    return this.name;
};

ScoredPropertyIntegerElement.prototype.createPrintTicket = function(rootNode, parentNode){
    var ret = false;
    try
    {
        var value = this.property.getCurrentValue( this.name );
        var node = null;
        if( (! this.property.isSupported(this.name) && value === null ) || value == this.value )
        {
            node = CreateScoredPropertyIntegerElement(
                            rootNode,
                            this.name.substring( this.name.indexOf(":") + 1),
                            this.value,
                            this.name.substring( 0, this.name.indexOf(":") )
                            );
        }

        if( node )
        {
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};


ScoredPropertyIntegerElement.prototype.createPrintCapabilities = function(rootNode, parentNode){
    var ret = false;
    try
    {
        var node = CreateScoredPropertyIntegerElement(
                        rootNode,
                        this.name.substring( this.name.indexOf(":") + 1),
                        this.value,
                        this.name.substring( 0, this.name.indexOf(":") )
                        );

        if( node )
        {
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};


ScoredPropertyIntegerElement.prototype.getName = function() {
    return this.name;
};


ScoredPropertyStringElement.prototype.createPrintTicket = function(rootNode, parentNode){
    var ret = false;
    try
    {
        var value = this.property.getCurrentValue( this.name );
        var node = null;
        if( value === null || value == this.value )
        {
            node = CreateScoredPropertyStringElement(
                            rootNode,
                            this.name.substring( this.name.indexOf(":") + 1),
                            this.value,
                            this.name.substring( 0, this.name.indexOf(":") )
                            );
        }

        if( node )
        {
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};


ScoredPropertyStringElement.prototype.createPrintCapabilities = function(rootNode, parentNode){
    var ret = false;
    try
    {
        var node = CreateScoredPropertyStringElement(
                        rootNode,
                        this.name.substring( this.name.indexOf(":") + 1),
                        this.value,
                        this.name.substring( 0, this.name.indexOf(":") )
                        );

        if( node )
        {
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

ScoredPropertyStringElement.prototype.getName = function() {
    return this.name;
};



var ParameterDefIntegerElement = function( property, name, value, unittype, mandatory, min, max, defaultvalue, multiple, display)
{
    this.property= property;
    this.name = name;
    this.value = value;
    this.unittype = unittype;
    this.mandatory = mandatory;
    this.min = min;
    this.max = max;
    this.defaultvalue = defaultvalue;
    this.multiple = multiple;
    this.display = display;
};

ParameterDefIntegerElement.prototype.createPrintTicket = function(rootNode, parentNode){
    var ret = false;
    try
    {
        removeChildNode( rootNode, "psf:PrintTicket/psf:ParameterInit[@name='" + this.name + "']" );

        {
            var node = CreateIntParamRefIni(
                    rootNode,
                    this.name.substring( this.name.indexOf(":") + 1),
                    (this.name.substring( 0, this.name.indexOf(":") ) == "psk" ),
                    "xsd:integer",
                    this.value
                    );

            if( node )
            {
                parentNode.appendChild( node );
                ret = true;
            }
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

ParameterDefIntegerElement.prototype.createPrintCapabilities = function(rootNode, parentNode){
    var ret = false;
    try
    {
        removeChildNode( rootNode, "psf:PrintCapabilities/psf:ParameterDef[@name='" + this.name + "']" );

        var integerdefault = this.min;
        if (this.defaultvalue !== null)
        {
            integerdefault = this.defaultvalue;
        }
        var node = CreateIntParamDef(
                rootNode,
                this.name.substring( this.name.indexOf(":") + 1),
                (this.name.substring( 0, this.name.indexOf(":") ) == "psk" ),
                integerdefault,
                this.min,
                this.max,
                this.multiple,
                this.unittype
                );

        if( node ) {
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

var ParameterDefStringElement = function( property, name, value, unittype, mandatory, min, max, defaultvalue, display)
{
    this.property= property;
    this.name = name;
    this.value = value;
    this.unittype = unittype;
    this.mandatory = mandatory;
    this.min = min;
    this.max = max;
    this.defaultvalue = defaultvalue;
    this.display = display;
};

ParameterDefStringElement.prototype.createPrintTicket = function(rootNode, parentNode){
    //debugger;
    var ret = false;
    try
    {
        removeChildNode( rootNode, "psf:PrintTicket/psf:ParameterInit[@name='" + this.name + "']" );

        {
            var node = CreateIntParamRefIni(
                    rootNode,
                    this.name.substring( this.name.indexOf(":") + 1),
                    (this.name.substring( 0, this.name.indexOf(":") ) == "psk" ),
                    "xsd:string",
                    this.value
                    );

            if( node )
            {
                parentNode.appendChild( node );
                ret = true;
            }
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};

ParameterDefStringElement.prototype.createPrintCapabilities = function(rootNode, parentNode){
    var ret = false;
    try
    {
        removeChildNode( rootNode, "psf:PrintCapabilities/psf:ParameterDef[@name='" + this.name + "']" );

        var node = CreateStringParamDef(
                rootNode,
                this.name.substring( this.name.indexOf(":") + 1),
                (this.name.substring( 0, this.name.indexOf(":") ) == "psk" ),
                this.defaultvalue,
                this.min,
                this.max,
                this.unittype
                );

        if( node ) {
            parentNode.appendChild( node );
            ret = true;
        }
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
};


var ModelSpec = function( modelSpecJson ) {
    this.modelSpec = modelSpecJson;

    this.getDefault = function( name ) {
        if( name in this.modelSpec && KEY_DEFAULT in this.modelSpec[name] )
        {
            return this.modelSpec[name][KEY_DEFAULT];
        }
        else
        {
            return null;
        }
    };

    this.getPickoneList = function( name ) {
        if( name in this.modelSpec )
        {
            return this.modelSpec[name][KEY_PICKONE];
        }
        else
        {
            return null;
        }
    };

    this.getMinValue = function( name ) {
        return this.modelSpec[name][KEY_MIN];
    };

    this.getMaxValue = function( name ) {
        return this.modelSpec[name][KEY_MAX];
    };

    this.isSupported = function( name ) {
        if( name in this.modelSpec ||
            containsInArray(name, AUTO_CREATE_FEATURE_LIST) ||
            containsInArray(name, AUTO_CREATE_PARAMETERDEF_LIST) )
        {
            return true;
        }
        else
        {
            return false;
        }
    };

    this.getFeatureNameList = function() {
        var list = [];
        for( var feature in this.modelSpec ) {
            list.push( feature );
        }
        return list;
    };

};


function removeChildNode( ptpc, xpath )
{
    var node = ptpc.XmlNode.selectSingleNode(xpath);
    if( node ) {
        ptpc.XmlNode.documentElement.removeChild( node );
    }
}

var Property = function( modelSpec, devmode, rcf, ptpsdMap ) {
    this.modelSpec = modelSpec;
    this.devmode   = devmode;
    this.rcf       = rcf;
    this.ptpsdMap  = ptpsdMap;
    this.refered   = [];

    this.getCurrentValue = function( name )
    {
        var result = null;
        if( this.rcf )
        {
            var psdkey = PtToPsdFeatureKey(name, this.ptpsdMap);
            var fixvalue = this.rcf.getFixValue( psdkey );
            if( fixvalue !== null)
            {
                result = PsdToPtValueKey( name, fixvalue, this.ptpsdMap );
            }

            if( result === null &&
                global.eventtype === EVENT_DEVMODETOPRINTTICKET &&
                global.isChangeDefault )
            {
                var rcfDefault =  this.rcf.getDefault( psdkey );
                if( rcfDefault )
                {
                    result = PsdToPtValueKey( name, rcfDefault, this.ptpsdMap );
                }
            }
        }

        if( result === null )
        {
            if( this.devmode && name in this.devmode )
            {
                result = this.devmode[name];
            }
            else
            {
                result = this.modelSpec.getDefault(name);
            }
        }

        if( result )
        {
            var pickoneList = this.getPickoneList( name );
            if( pickoneList &&
                !containsInArray(result, pickoneList) &&
                !containsInArray(name, CHANGEABLE_PICKONE_FEATURE_LIST ) )
            {
                result = this.modelSpec.getDefault( name );
            }
        }

        return result;
    };

    this.setCurrentValue = function( name, value ) {
        var result = false;
        if( value != null )
        {
            if( this.rcf )
            {
                var fixvalue = this.rcf.getFixValue( PtToPsdFeatureKey(name, this.ptpsdMap) );
                if( fixvalue !== null)
                {
                    this.devmode[name] = PsdToPtValueKey( name, fixvalue, this.ptpsdMap );
                    result = true;
                }
            }

            if( !result ) {
                this.devmode[name] = value;
            }
        }
    };

    this.getDefault = function( name )
    {
        var rcfDefault = this.getCustomizedDefault( name );

        return (rcfDefault) ? rcfDefault : this.modelSpec.getDefault(name);
    };

    this.getCustomizedDefault = function( name )
    {
        if( this.rcf )
        {
            var rcfdefault = this.rcf.getDefault( PtToPsdFeatureKey(name, this.ptpsdMap) );
            if( rcfdefault !== null )
            {
                return PsdToPtValueKey(name, rcfdefault, this.ptpsdMap);
            }
        }
        return null;
    };

    this.getPickoneList = function( name )
    {
        if( this.rcf )
        {
            var psdKey = PtToPsdFeatureKey(name, this.ptpsdMap);
            var rcfpickone = this.rcf.getPickoneList( psdKey );
            if( rcfpickone )
            {
                var pickone = [];
                for(var i=0; i<rcfpickone.length; i++ )
                {
                    pickone[i] = PsdToPtValueKey( name, rcfpickone[i], this.ptpsdMap );
                }
                if( pickone.length > 0 )
                {
                    return pickone;
                }
            }

            var fixvalue = this.rcf.getFixValue( psdKey );
            if( fixvalue )
            {
                return [ PsdToPtValueKey( name, fixvalue, this.ptpsdMap ) ];
            }
        }

        return this.modelSpec.getPickoneList(name);
    };

    this.getMinValue = function( name ) {
        return this.modelSpec.getMinValue(name);
    };

    this.getFixValue = function( name ) {
        var result = null;
        if( this.rcf )
        {
            var fixvalue = this.rcf.getFixValue( PtToPsdFeatureKey(name, this.ptpsdMap) );
            if( fixvalue )
            {
                result = PsdToPtValueKey( name, fixvalue, this.ptpsdMap );
            }
        }
        return result;
    };

    this.getMaxValue = function( name ) {
        var result = null;
        if( this.rcf )
        {
            var psdKey = PtToPsdFeatureKey(name, this.ptpsdMap);
            result = this.rcf.getMaxValue( psdKey );
        }
        
        if( !result )
        {
            result = this.modelSpec.getMaxValue(name);
        }
        return result;
    };

    this.isSupported = function( name ) {
        return this.modelSpec.isSupported(name);
    };

    this.getFeatureNameList = function() {
        return this.modelSpec.getFeatureNameList();
    };

    this.setReferedParameterDef = function( name ) {
        return this.refered.push(name);
    };

    this.isReferedParameterDef = function( name ) {
        return containsInArray( name, this.refered );
    };
};

//
// RCF̏Ɋւ
//
var RCF = function( rcfDom, ptpsdMap ) {
    this.dom       = rcfDom;
    this.rcfInfo   = {};
    this.initialize();
};

RCF.prototype.initialize = function()
{
    try
    {
        var version = RCFParser_GetVersion( this.dom );
        
        if( version == "1.0" )
        {
            var customer       = RCFParser_GetCustomer( this.dom );
            var devicesettings = RCFParser_GetDeviceSettings( this.dom);
            var featurelock    = RCFParser_GetFeatureLock( this.dom );
            var modifyitem     = RCFParser_GetModifyItem( this.dom );

            var item = {};
            if( devicesettings[RCF_ITEM_KEY_ITEMLIST] )
            {
                for(var i=0; i<devicesettings[RCF_ITEM_KEY_ITEMLIST].length; i++ )
                {
                    item = {};
                    item[RCF_ITEM_KEY_DEFAULT] = devicesettings[RCF_ITEM_KEY_ITEMLIST][i][RCF_ITEM_KEY_VALUE];
                    this.rcfInfo[ devicesettings[RCF_ITEM_KEY_ITEMLIST][i][RCF_ITEM_KEY_NAME] ] = item;
                }
            }

            for(var i=0; i<featurelock.length; i++ )
            {
                item = {};
                if( featurelock[i][RCF_ITEM_KEY_NAME] in this.rcfInfo )
                {
                    item = this.rcfInfo[ featurelock[i][RCF_ITEM_KEY_NAME] ];
                }
                item[RCF_ITEM_KEY_FIXVALUE] = featurelock[i][RCF_ITEM_KEY_FIXVALUE];
                item[RCF_ITEM_KEY_PICKONE]  = featurelock[i][RCF_ITEM_KEY_PICKONE];

                this.rcfInfo[ featurelock[i][RCF_ITEM_KEY_NAME] ] = item;
            }
            
            for(var i=0; i<modifyitem.length; i++)
            {
                item = {};
                if( modifyitem[i][RCF_ITEM_KEY_NAME] in this.rcfInfo )
                {
                    item = this.rcfInfo[ modifyitem[i][RCF_ITEM_KEY_NAME] ];
                }
                item[RCF_ITEM_KEY_MAX] = modifyitem[i][RCF_ITEM_KEY_MAX];

                this.rcfInfo[ modifyitem[i][RCF_ITEM_KEY_NAME] ] = item;
            }


        }
    }
    catch(e)
    {
        this.rcfInfo = {};
    }
};

RCF.prototype.getDefault = function( name )
{
    var result = null;
    if( name in this.rcfInfo )
    {
        if( RCF_ITEM_KEY_DEFAULT in this.rcfInfo[name] )
        {
            result = this.rcfInfo[name][RCF_ITEM_KEY_DEFAULT];
        }
        else if( RCF_ITEM_KEY_FIXVALUE in this.rcfInfo[name] )
        {
            result = rcfInfo[name][RCF_ITEM_KEY_FIXVALUE];
        }
        else if( RCF_ITEM_KEY_PICKONE in this.rcfInfo[name] &&
                 this.rcfInfo[name][RCF_ITEM_KEY_PICKONE].length == 1 )
        {
            result = this.rcfInfo[name][RCF_ITEM_KEY_FIXVALUE][0];
        }
        else
        {
            result = null;
        }
    }
    return result;
};

RCF.prototype.getFixValue = function( name )
{
    var result = null;
    if( name in this.rcfInfo )
    {
        if( RCF_ITEM_KEY_FIXVALUE in this.rcfInfo[name] )
        {
            result = this.rcfInfo[name][RCF_ITEM_KEY_FIXVALUE];
        }
        else if(    RCF_ITEM_KEY_PICKONE in this.rcfInfo[name] &&
                    this.rcfInfo[name][RCF_ITEM_KEY_PICKONE].length == 1 )
        {
            result = this.rcfInfo[name][RCF_ITEM_KEY_PICKONE][0];
        }
        else
        {
            result = null;
        }
    }
    return result;
};

RCF.prototype.getPickoneList = function( name )
{
    var result = null;
    if(    name in this.rcfInfo && RCF_ITEM_KEY_PICKONE in this.rcfInfo[name] )
    {
        result = this.rcfInfo[name][RCF_ITEM_KEY_PICKONE];
    }
    else
    {
        result = null;
    }
    return result;
};

RCF.prototype.getMaxValue = function( name )
{
    var result = null;
    if( name in this.rcfInfo )
    {
        if( RCF_ITEM_KEY_MAX in this.rcfInfo[name] )
        {
            result = this.rcfInfo[name][RCF_ITEM_KEY_MAX];
        }
    }
    return result;
};


function getDevmodeObject( devModeProperties )
{
    var settingsObject = null;
    try
    {
        var settings = devModeProperties.GetString("DocumentSettings");

        settingsObject = (new Function("return " + settings))();
    }
    catch(e)
    {
        settingsObject = null;
    }

    return settingsObject;
}


function getInnerSpecObject( scriptContext )
{
    var innerSpec = scriptContext.DriverProperties.GetString("pc_table");

    var innerSpecObject = ConvertToJsonObjectFromString(innerSpec);

    return innerSpecObject;
}


function getModelSpecObject( scriptContext, modelName )
{
    var modelSpec = GetModelSpecTable(scriptContext, modelName);
    var modelSpecJson = ConvertToJsonObjectFromString(modelSpec);
    var modelSpecObject = new ModelSpec( modelSpecJson );

    return modelSpecObject;
}


function getXPathTableObject(scriptContext) {

    var str = scriptContext.DriverProperties.GetString("pt_xpath");
    var xpathTable = ConvertToJsonObjectFromString(str);

    return xpathTable;
}


function getPtPsdMapObject( scriptContext )
{
    var pt_table = scriptContext.DriverProperties.GetString("pt_table");
    var tableObj = ConvertToJsonObjectFromString(pt_table);

    return tableObj;
}


function createAllPrintTicket(devModeProperties, scriptContext, printTicket)
{
    try {
        var modelName = GetCurrentModelName(scriptContext);
        if( modelName !== null )
        {
            var property = createPropertyObject(scriptContext, devModeProperties, printTicket );

            var innerSpecObject = getInnerSpecObject( scriptContext );
            var table = getPrintTicketTreeTable( innerSpecObject, property, scriptContext );
            for( var i=0; i<table.length; i++ )
            {
                table[i].createPrintTicket(
                            printTicket, 
                            printTicket.XmlNode.lastChild
                            );
            }

            updateResolutionDevmodeValue( printTicket, property);

            if( isEnableCustomization(scriptContext) )
            {
                updateDefaultCustomizePrintTicket( printTicket, property );
            }
        }
    }
    catch (e) {
        var result = 0;
        result = e.number;
        Debug.write(e.message + "createAllPrintTicket() Error!");
    }
}

function updateDefaultCustomizePrintTicket( printTicket, property )
{
    var xpath = null;
    var customized = null;
    for(var j=0; j<AUTO_CREATE_FEATURE_LIST.length; j++)
    {
        customized = property.getCustomizedDefault( AUTO_CREATE_FEATURE_LIST[j] );
        if( customized )
        {
            xpath = "psf:Feature[@name='" + AUTO_CREATE_FEATURE_LIST[j] + "']/psf:Option";
            var node =  printTicket.XmlNode.documentElement.selectSingleNode( xpath );
            if( node )
            {
                node.attributes[0].text = customized;
            }
        }
    }

    for(var j=0; j<AUTO_CREATE_PARAMETERDEF_LIST.length; j++)
    {
        customized = property.getCustomizedDefault( AUTO_CREATE_PARAMETERDEF_LIST[j] );
        if( customized )
        {
            xpath = "psf:ParameterInit[@name='" + AUTO_CREATE_PARAMETERDEF_LIST[j] + "']/psf:Value";
            var node =  printTicket.XmlNode.documentElement.selectSingleNode( xpath );
            if( node )
            {
                node.text = customized;
            }
        }
    }

    for(var j=0; j<AUTO_CREATE_SCOREDPROPERTY_LIST.length; j++)
    {
        customized = property.getCustomizedDefault( AUTO_CREATE_SCOREDPROPERTY_LIST[j] );
        if( customized )
        {
            xpath = "psf:Feature/psf:Option/psf:ScoredProperty[@name='" + AUTO_CREATE_SCOREDPROPERTY_LIST[j] + "']/psf:Value";
            var node =  printTicket.XmlNode.documentElement.selectSingleNode( xpath );
            if( node )
            {
                node.text = customized;
            }
        }
    }

}

function updateResolutionDevmodeValue(printTicket, property) {
    var xpath = null;

    try
    {
        xpath = "psf:Feature[@name='psk:PageResolution']/psf:Option";
        var resolution = printTicket.XmlNode.documentElement.selectSingleNode(xpath);

        if ( (resolution) && (resolution.childNodes.length != 0) )
        {
            if ((RESOLUTION_MAP[property.devmode['psk:PageResolution']]['ResolutionX'] == resolution.childNodes[0].text) && (RESOLUTION_MAP[property.devmode['psk:PageResolution']]['ResolutionY'] == resolution.childNodes[1].text))
            {
                resolution.attributes[0].text = property.devmode['psk:PageResolution'];
            }
        }
    }
    catch (e)
    {
        Debug.write(e.message + "updateResolutionDevmodeValue() Error!");
    }
}

function convertAllFeaturesPrintTicketToDevmode(printTicket, scriptContext, devModeProperties)
{
    try
    {
        global.eventtype = EVENT_PRINTTICKETTODEVMODE;

        var property = createPropertyObject( scriptContext, devModeProperties, printTicket );

        var xpathTable = getXPathTableObject( scriptContext );

        if( property && xpathTable )
        {
            var printTicketParser = new PrintTicketParser( printTicket, xpathTable );
            var features = property.getFeatureNameList();
            for(var i=0; i<features.length; i++ )
            {
                var xpath = getXPathString(features[i], xpathTable);
                if( xpath !== null )
                {
                    var node = printTicket.XmlNode.documentElement.selectSingleNode( xpath );
                    if( containsInArray( features[i], AUTO_CREATE_FEATURE_LIST ) ||
                        containsInArray( features[i], AUTO_CREATE_PARAMETERDEF_LIST ) )
                    {
                        var fixvalue = property.getFixValue(features[i]);
                        if( fixvalue )
                        {
                            if( node !== null )
                            {
                                node.text = fixvalue;
                            }
                        }
                    }
                    else
                    {
                        if( node !== null )
                        {
                            property.setCurrentValue( features[i], node.text );
                        }
                        else
                        {
                            property.setCurrentValue( features[i], printTicketParser.getValue( features[i] ) );
                        }
                    }
                }
            }

            var newsetting = convertJsonObjectToString( property.devmode );

            devModeProperties.SetString("DocumentSettings", newsetting);
        }
    }
    catch(e)
    {

    }
    return;
}

function validateDevmodeToPrintTicket( printTicket, scriptContext )
{
    var xpathTable = getXPathTableObject( scriptContext );

    var printTicketParser = new PrintTicketParser( printTicket, xpathTable );

    validateDuplexPrintTicket( printTicket );

    validateOutputColorPrintTicket( printTicket, printTicketParser );

    validateCustomPaperSizePrintTicket( printTicket, printTicketParser );
}

function validatePrintTicketToDevmode( printTicket, scriptContext )
{
    var xpathTable = getXPathTableObject( scriptContext );

    var printTicketParser = new PrintTicketParser( printTicket, xpathTable );

    validatePaperSizePrintTicket( printTicket );

    validateInputBinPrintTicket( printTicket );

    validateDuplexPrintTicket( printTicket );

    validateResolutionPrintTicket( printTicket );

    validateOutputColorPrintTicket( printTicket, printTicketParser );

    validateCustomPaperSizePrintTicket( printTicket, printTicketParser );
}

function validatePaperSizePrintTicket( printTicket )
{
    var xpath = "psf:Feature[@name='psk:PageMediaSize']/psf:Option[@name='ns0000:FOLIO']";
    var folio = printTicket.XmlNode.documentElement.selectSingleNode( xpath );
    if (folio)
    {
        folio.attributes[0].text = "psk:OtherMetricFolio";
    }
}

function validateCustomPaperSizePrintTicket( printTicket, printTicketParser )
{
    var documentSize = printTicketParser.getValue( "psk:PageMediaSize" );
    var documentSizeCustomWidth  = printTicketParser.getValue( "psk:PageMediaSizeMediaSizeWidth" );
    var documentSizeCustomHeight = printTicketParser.getValue( "psk:PageMediaSizeMediaSizeHeight" );

    var printSizeCustomWidth  = printTicketParser.getValue( "ns0000:PageMediaSizeMediaSizeWidth" );
    var printSizeCustomHeight = printTicketParser.getValue( "ns0000:PageMediaSizeMediaSizeHeight" );

    if( documentSize === "psk:CustomMediaSize" &&
        documentSizeCustomWidth !== null && documentSizeCustomHeight !== null &&
        printSizeCustomWidth    !== null && printSizeCustomHeight    !== null )
    {
        var printWidthNode  = printTicketParser.getNode( "ns0000:PageMediaSizeMediaSizeWidth" );
        var printHeightNode = printTicketParser.getNode( "ns0000:PageMediaSizeMediaSizeHeight" );

        printWidthNode.text  = documentSizeCustomWidth;
        printHeightNode.text = documentSizeCustomHeight;
    }
}


function validateInputBinPrintTicket( printTicket )
{
    var xpath = null;
    
    xpath = "psf:Feature[@name='psk:JobInputBin']";
    var inputbin = printTicket.XmlNode.documentElement.selectSingleNode( xpath );
    if (inputbin)
    {
        inputbin.attributes[0].text = "psk:PageInputBin";
    }

    xpath = "psf:Feature[@name='psk:PageInputBin']/psf:Option[@name='psk:AutoSelect']";
    var autoSelect = printTicket.XmlNode.documentElement.selectSingleNode( xpath );
    if (autoSelect)
    {
        autoSelect.attributes[0].text = "ns0000:AUTO";
    }
}

function validateDuplexPrintTicket( printTicket )
{
    var xpath = null;

    xpath = "psf:Feature[@name='psk:JobDuplexAllDocumentsContiguously']/psf:Option";
    var contiguously = printTicket.XmlNode.documentElement.selectSingleNode(xpath);

    xpath = "psf:Feature[@name='ns0000:JobDuplexAllDocumentsDirection']/psf:Option";
    var direction = printTicket.XmlNode.documentElement.selectSingleNode(xpath);
    if (!direction)
    {
        var directionnode = printTicket.XmlNode.documentElement.selectNodes("psf:Feature");
        for (var i = 0; i < directionnode.length; i++)
        {
            if (directionnode[i].attributes[0].text == "ns0000:JobDuplexAllDocumentsDirection")
            {
                direction = directionnode[i].selectSingleNode("psf:Option");
                break;
            }
        }
    }

    if ((contiguously) && (direction))
    {
        if (contiguously.attributes[0].text != PSKJOBDUPLEX_CONTIGUOUSLY_MAP[direction.attributes[0].text])
        {
            direction.attributes[0].text = RPSKJOBDUPLEX_DIRECTION_MAP[contiguously.attributes[0].text][direction.attributes[0].text];
        }
    }
}

function validateResolutionPrintTicket(printTicket) {
    var xpath = null;

    try
    {
        xpath = "psf:Feature[@name='psk:PageResolution']/psf:Option";
        var resolution = printTicket.XmlNode.documentElement.selectSingleNode(xpath);

        if ( (resolution) && (resolution.childNodes.length != 0) )
        {
            if ((RESOLUTION_MAP[resolution.attributes[0].text]['ResolutionX'] != resolution.childNodes[0].text) && (RESOLUTION_MAP[resolution.attributes[0].text]['ResolutionY'] != resolution.childNodes[1].text))
            {
                resolution.childNodes[0].text = RESOLUTION_MAP[resolution.attributes[0].text]['ResolutionX'];
                resolution.childNodes[1].text = RESOLUTION_MAP[resolution.attributes[0].text]['ResolutionY'];
            }
        }
    }
    catch(e)
    {
        Debug.write(e.message + "validateResolutionPrintTicket() Error!");
    }
} 


function validateOutputColorPrintTicket(printTicket, printTicketParser) {
    try
    {
        var publicColor  = printTicketParser.getValue( "psk:PageOutputColor" );
        var privateColor = printTicketParser.getValue( "ns0000:PageDriverOutputColor" );
        var fixdmColor   = printTicketParser.getValue( "ns0000:JobFixdmColor" );

        if( publicColor && privateColor && fixdmColor )
        {
            if( fixdmColor === "ns0000:Off" )
            {
                xpath = "psf:Feature[@name='ns0000:PageDriverOutputColor']";
                var outputcolor = printTicket.XmlNode.documentElement.selectSingleNode(xpath);
                if( !outputcolor )
                {
                    var outputcolor_node = printTicket.XmlNode.documentElement.selectNodes( "psf:Feature" );
                    for ( var i = 0; i < outputcolor_node.length; i++) {
                       if ( outputcolor_node[i].attributes[0].text == "ns0000:PageDriverOutputColor" ) {
                            outputcolor = outputcolor_node[i];
                            break;
                        }
                    }
                }

                if( outputcolor ) {
                    printTicket.XmlNode.documentElement.removeChild( outputcolor );
                }
            }
            else if( fixdmColor === "ns0000:On" )
            {
                var psknode = printTicketParser.getNode( "psk:PageOutputColor" );
                psknode.text = "psk:Color";
            }
        }
    }
    catch(e)
    {
        Debug.write(e.message + "validateOutputColorPrintTicket() Error!");
    }
}


function removePrefix( withPrefix )
{
    return withPrefix.substring( withPrefix.indexOf(":") + 1 );
}

function createPropertyObject( scriptContext, devModeProperties, ptpc )
{
    var property = null;
    try
    {
        if( !scriptContext ) {
            return null;
        }

        var devmodeObject = null;
        if( devModeProperties ) {
            devmodeObject = getDevmodeObject( devModeProperties );
            if( !devmodeObject ) {
                devmodeObject = {};
            }
        }
 
        var modelName = GetCurrentModelName(scriptContext);
        var modelSpecObject = getModelSpecObject( scriptContext, modelName );

        var uiCustom = scriptContext.QueueProperties.GetString("UI_Customize");
        var rcf = null;
        var ptpsdMap = null;
        if( uiCustom.length > 0 )
        {
            var rcfDom = CreateDomDoc( ptpc );
            if( rcfDom.loadXML(uiCustom) )
            {
                rcf = new RCF( rcfDom );
                ptpsdMap = getPtPsdMapObject( scriptContext );
            }
        }

        property = new Property(    modelSpecObject,
                                        devmodeObject,
                                        rcf,
                                        ptpsdMap,
                                        (devModeProperties) ? true : false );
    }
    catch(e)
    {
        property = null;
        Debug.write(e.message + "!!! createPropertyObject() Failed !!!");
    }

    return property;
}

function convertJsonObjectToString( json )
{
    var jsonstr = "";
    if( json && json != {} )
    {
        jsonstr = "{";
        for(var key in json )
        {
            jsonstr += "\"" + key + "\":\"" + json[key] + "\",";
        }
        jsonstr = jsonstr.substr(0, jsonstr.length-1 );
        jsonstr += "}";
    }

    return jsonstr;
}

function getXPathFromPrintSchemaTable(featureName, ps_table)
{
    var ret = null;
    for(var i=0; i<ps_table.length; i++)
    {
        if( featureName == ps_table[i][KEY_ATTRIBUTE] )
        {
            if( "xpath" in ps_table[i] ) {
                ret = ps_table[i]["xpath"];
            }
            break;
        }
    }
    return ret;
}

function getXPathString(feature, xpathTable)
{
    var ret = null;
    if (feature in xpathTable)
    {
        if (xpathTable[feature] === 0)
        {
            ret = null;
        }
        else if (xpathTable[feature] == 1)
        {
            ret = "psf:Feature[@name='" + feature + "']/psf:Option/@name";
        }
        else if (xpathTable[feature] == 2)
        {
            ret = "psf:ParameterInit[@name='" + feature + "']/psf:Value";
        }
        else if ( "3" in xpathTable[feature] )
        {
            for(var i=0; i<xpathTable[feature]["3"].length; i++ )
            {
                if (ret === null)
                {
                    ret = "psf:Feature[@name='" + xpathTable[feature]["3"][i] + "']/";
                }
                else
                {
                    ret += "psf:Feature[@name='" + xpathTable[feature]["3"][i] + "']/";
                }
            }
            ret += "psf:Feature[@name='" + feature + "']/psf:Option/@name";
        }
        else if ( "4" in xpathTable[feature] )
        {
            ret = "psf:Feature[@name='" + xpathTable[feature]["4"][0] + "']/psf:Option/psf:ScoredProperty[@name='" + feature + "']/psf:Value";
        }
        else
        {
        }
    }
    else
    {
        ret = null;
    }
    
    return ret;
}

function WideCharBytesToMultiCharByte( wideCharBytes )
{
    var multi = [];
    for(var i=0, j=0; i<wideCharBytes.length; i=i+2, j++)
    {
        multi[j] = (wideCharBytes[i + 1] << 8) + wideCharBytes[i];
    }
    return multi;
}

function ConvertToUTF8Bytes(bytes) {
    var byteUTF8 = [];
    for (var i = 0, j = 0; i < bytes.length; i++, j++) {
        if (bytes[i] <= 0x7f) {
            byteUTF8[j] = bytes[i];
        } else if (bytes[i] <= 0xdf) {
            byteUTF8[j] = ((bytes[i] & 0x1f) << 6);
            byteUTF8[j] += bytes[i + 1] & 0x3f;
            i++;
        } else if (bytes[i] <= 0xe0) {
            byteUTF8[j] = ((bytes[i + 1] & 0x1f) << 6) | 0x0800;
            byteUTF8[j] += bytes[i + 2] & 0x3f;
            i += 2;
        } else {
            byteUTF8[j] = ((bytes[i] & 0x0f) << 12);
            byteUTF8[j] += (bytes[i + 1] & 0x3f) << 6;
            byteUTF8[j] += bytes[i + 2] & 0x3f;
            i += 2;
        }
    }
    return byteUTF8;
}

function PtToPsdFeatureKey( ptKey, ptpsdTable )
{
    for( var j=0; j<ptpsdTable[PTPSD_KEY_ATTRIBUTE].length; j++)
    {
        if( ptKey == ptpsdTable[PTPSD_KEY_ATTRIBUTE][j][PTPSD_KEY_PTNAME] )
        {
            return ptpsdTable[PTPSD_KEY_ATTRIBUTE][j][PTPSD_KEY_PSDNAME];
        }
    }

    return null;
}

function PsdToPtValueKey( ptFeature, psdValue, ptpsdTable )
{
    var result = null;
    for( var i=0; i<ptpsdTable[PTPSD_KEY_ATTRIBUTE].length; i++)
    {
        if( ptFeature == ptpsdTable[PTPSD_KEY_ATTRIBUTE][i][PTPSD_KEY_PTNAME] )
        {
            if( "pickone" in ptpsdTable[PTPSD_KEY_ATTRIBUTE][i] )
            {
                for( var j=0; j<ptpsdTable[PTPSD_KEY_ATTRIBUTE][i][RCF_ITEM_KEY_PICKONE].length; j++ )
                {
                    if( ptpsdTable[PTPSD_KEY_ATTRIBUTE][i][PTPSD_KEY_PICKONE][j][PTPSD_KEY_PSDNAME] == psdValue )
                    {
                        result = ptpsdTable[PTPSD_KEY_ATTRIBUTE][i][PTPSD_KEY_PICKONE][j][PTPSD_KEY_PTNAME];
                        break;
                    }
                }
            }
            else
            {
                result = psdValue;
            }
            break;
        }
    }
    return result;
}

function RCFParser_GetVersion(rcf)
{
    var version = null;
    try
    {
        var root = rcf.selectSingleNode("rcf");
        version = root.getAttribute("version");
    }
    catch(e)
    {
    }
    return version;
}

function RCFParser_GetCustomer(rcf)
{
    var customer = { "id": "" };
    try
    {
        var customerNode = rcf.selectSingleNode("rcf/customer");
        var id = customerNode.getAttribute("id");
        customer['id'] = id;
    }
    catch (e) {
        customer = null;
    }
    return customer;
}

function RCFParser_GetDeviceSettings(rcf)
{
    var devicesettings = { "drivername": "", "independent":"no", RCF_ITEM_KEY_ITEMLIST: [] };

    try {
        var devicesettingsNode = rcf.selectSingleNode("rcf/devicesettings");
        devicesettings['drivername'] = devicesettingsNode.getAttribute("drivername");
        devicesettings['independent'] = devicesettingsNode.getAttribute("independent");

        var itemNodes = rcf.selectNodes("rcf/devicesettings/item");
        var ary = [];
        for( var i=0; i < itemNodes.length; i++)
        {
            var item = { RCF_ITEM_KEY_NAME: "", RCF_ITEM_KEY_VALUE: "" };
            item[RCF_ITEM_KEY_NAME] = itemNodes[i].getAttribute("name");
            item[RCF_ITEM_KEY_VALUE]= itemNodes[i].getAttribute("value");
            ary[i] = item;
        }
        devicesettings[RCF_ITEM_KEY_ITEMLIST] = ary;

    }
    catch (e) {
    }
    return devicesettings;
}

function RCFParser_GetFeatureLock(rcf)
{
    var lockItems = [];
    try {
        var featureLockNode = rcf.selectSingleNode("rcf/featurelock");

        var itemNodes = rcf.selectNodes("rcf/featurelock/item");
        for (var i = 0; i < itemNodes.length; i++) {

            var item = { RCF_ITEM_KEY_NAME: "", RCF_ITEM_KEY_FIXVALUE: null, RCF_ITEM_KEY_PICKONE: null };
            item[RCF_ITEM_KEY_NAME] = itemNodes[i].getAttribute("name");
            item[RCF_ITEM_KEY_FIXVALUE] = itemNodes[i].getAttribute("fixvalue");
            if (item[RCF_ITEM_KEY_FIXVALUE] === null)
            {
                var pickoneNodes = itemNodes[i].selectNodes("pickone");
                var pickone = [];
                for (var j = 0; j < pickoneNodes.length; j++)
                {
                    pickone[j] = pickoneNodes[j].getAttribute("name");
                }
                item[RCF_ITEM_KEY_PICKONE] = pickone;
            }
            lockItems[i] = item;
        }
        
    }
    catch (e) {
    }
    return lockItems;
}

function RCFParser_GetModifyItem(rcf)
{
    var modifyItems = [];
    try {
        var itemNodes = rcf.selectNodes("rcf/modifyitem/item");

        for (var i = 0; i < itemNodes.length; i++) {

            var item = {};
            var name = itemNodes[i].getAttribute("name");
            var max  = null;
            var node  = itemNodes[i].selectSingleNode("string");
            if( node )
            {
                max  = node.getAttribute("max");
            }
            if( max && name)
            {
                item[RCF_ITEM_KEY_MAX]  = max;
                item[RCF_ITEM_KEY_NAME] = name;
                modifyItems.push( item );
            }
        }
    }
    catch (e) {
    }
    return modifyItems;
}


function CreateDomDoc(ptpc)
{
    var domDoc = ptpc.XmlNode.cloneNode(false);
    if (domDoc.parseError.errorCode !== 0) {
       var myErr = workNode.parseError;
       Debug.write("You have error " + myErr.reason);
    } else {
       Debug.write(domDoc.xml);
    }
    return domDoc;
}

function CreateAllPrintCapabilities(printTicket, scriptContext, printCapabilities)
{
    try {
        var modelName = GetCurrentModelName(scriptContext);
        if( modelName !== null )
        {
            var commonPCList  = getInnerSpecObject( scriptContext );

            var modelSpecObject = getModelSpecObject( scriptContext, modelName );


            var uiCustom = scriptContext.QueueProperties.GetString("UI_Customize");
            var rcf = null;
            var ptpsdMap = null;
            if( uiCustom.length > 0 )
            {
                var rcfDom = CreateDomDoc(printCapabilities);
                if( rcfDom.loadXML(uiCustom) )
                {
                    rcf = new RCF( rcfDom );
                    ptpsdMap = getPtPsdMapObject( scriptContext );
                }
            }

            var property = new Property(modelSpecObject, null, rcf, ptpsdMap);
            var table = getPrintTicketTreeTable( commonPCList, property, scriptContext );
            for( var i=0; i<table.length; i++ )
            {
                table[i].createPrintCapabilities(
                            printCapabilities, 
                            printCapabilities.XmlNode.lastChild
                            );
            }
        }
    }
    catch (e) {

        var result = 0;
        result = e.number;
        Debug.write(e.message + "completePrintCapabilities() Error!");        
    }
    
    return;
}

var PrintTicketParser = function( printTicket, xpathTable )
{
    this.printTicket = printTicket;
    this.xpathTable  = xpathTable;
    this.valueList   = {};
    this.nodeList    = {};
    this.parse();
};

PrintTicketParser.prototype.parse = function() {
    var featureNodes = this.printTicket.XmlNode.documentElement.selectNodes("psf:Feature");
    try
    {
        for (var i = 0; i < featureNodes.length; i++)
        {
            var parentFeatureNameList = [];
            this.parseFeatureNode( featureNodes[i], parentFeatureNameList );
        }

        var parameterInitNodes = this.printTicket.XmlNode.documentElement.selectNodes("psf:ParameterInit");
        for (var i = 0; i < parameterInitNodes.length; i++)
        {
            var parameterName = parameterInitNodes[i].attributes[0].text;
            if (this.xpathTable[ parameterName ] == 2)
            {
                var value = parameterInitNodes[i].selectSingleNode("psf:Value");
                this.valueList[parameterName] = value.text;
                this.nodeList[parameterName]  = value;
            }
        }
    }
    catch(e)
    {
        Debug.write(e.message + "PrintTicketParser::parse() Error!");
    }
};

PrintTicketParser.prototype.parseFeatureNode = function( featureNode, parentFeatureNameList ) {

    try
    {
        var featureName = featureNode.attributes[0].text;

        var option = null;
        if( featureName in this.xpathTable )
        {
            if (this.xpathTable[ featureName ] == 1 && parentFeatureNameList.length == 0)
            {
                option = featureNode.selectSingleNode("psf:Option");
                this.valueList[featureName] = option.attributes[0].text;
                this.nodeList[featureName]  = option.attributes[0];
            }
            else if ( "3" in this.xpathTable[featureName] &&
                      parentFeatureNameList.length === this.xpathTable[featureName]['3'].length )
            {
                var match = true;
                for(var i=0; i<parentFeatureNameList.length; i++)
                {
                    if( parentFeatureNameList[i] !== this.xpathTable[featureName]['3'][i] )
                    {
                        match = false;
                        break;
                    }
                }

                if( match === true )
                {
                    option = featureNode.selectSingleNode("psf:Option");
                    this.valueList[featureName] = option.attributes[0].text;
                    this.nodeList[featureName]  = option.attributes[0];
                }
            }
        }

        var childFeatureNodes = featureNode.selectNodes("psf:Feature");
        parentFeatureNameList.push( featureName );
        for (var j = 0; j < childFeatureNodes.length; j++)
        {
            var parentSubFeatureNameList = parentFeatureNameList.concat();
            this.parseFeatureNode( childFeatureNodes[j], parentSubFeatureNameList );
        }
    }
    catch(e)
    {
        Debug.write(e.message + "PrintTicketParser::parseFeatureNode() Error : " + featureName);
    }
};

PrintTicketParser.prototype.getValue = function( name ) {
    if( name in this.valueList )
    {
        return this.valueList[name];
    }
    else
    {
        return null;
    }
};

PrintTicketParser.prototype.getNode = function( name ) {
    if( name in this.nodeList )
    {
        return this.nodeList[name];
    }
    else
    {
        return null;
    }
};

function containsInArray( value, arr )
{
    for(var i in arr)
    {
        if( arr.hasOwnProperty(i) && arr[i] === value){
            return true;
        }
    }
    return false;
}


function CreateElement( xmlNode, tag, tagNs )
{
    return xmlNode.createNode(1, tag, tagNs);
}

function CreateAttribute(xmlNode, attributeValue)
{
    var attributeNode = null;
    if (attributeValue !== "")
    {
        attributeNode = xmlNode.createNode(2, GetAttributeName(attributeValue), GetAttributeNameSpace(attributeValue));
        if (attributeNode) {
            attributeNode.value = attributeValue;
        }
    }
    return attributeNode;
}

function GetAttributeName( attributeValue )
{
    if( attributeValue.indexOf("xsd") === 0 )
    {
        return "xsi:type";
    }
    else
    {
        return "name";
    }
}

function GetAttributeNameSpace(attributeValue) {
    if (attributeValue.indexOf("xsd") === 0) {
        return SCHEMA_INST_URI;
    }
    else {
        return FRAMEWORK_URI;
    }
}

function GetPrefixURI(prefix)
{
    if( prefix == "psk")
    {
        return  pskNs;
    }
    else
    {
        return ricohNs;
    }
    return null;
}



function GetCurrentModelName(scriptContext)
{
    var model = null;
    try {
        model = scriptContext.QueueProperties.GetString("PrinterModelName");
        if (model == "auto") {
            model = "Generic Model";
        }
    }
    catch (e) {
        model = "Generic Model";
        var result = 0;
        result = e.number;
        Debug.write(e.message + "GetCurrentModelName() Error!");
    }

    return model;
}

function GetModelSpecTable(scriptContext, modelName)
{
    var model_map     = scriptContext.DriverProperties.GetString("modeldata_map");
    var modelDataList = ConvertToJsonObjectFromString(model_map);
    
    var modelSpecName = null;
    for (var i = 0; i < modelDataList.length; i++) {
        if( modelDataList[i]["model"] == modelName) {
            modelSpecName = modelDataList[i]["pctable"];
            break;
        }
    }

    return scriptContext.DriverProperties.GetString( modelSpecName );
}

function GetBytesFromProperties( properties, name)
{
    var bytes = null;
    try
    {
        bytes = properties.GetBytes(name);
    }
    catch (e) {
        var result = 0;
        result = e.number;
        Debug.write(e.message + "GetBytesFromProperties() Error!");
    }
    return bytes;
}


function GetBoolFromProperties( properties, name)
{
    var value = null;
    try
    {
        value = properties.GetBool(name);
    }
    catch (e) {
        var result = 0;
        result = e.number;
        Debug.write(e.message + "GetBytesFromProperties() Error!");
    }
    return value;
}



function ConvertToJsonObjectFromBytes(bytes)
{
    var byteString = String.fromCharCode.apply(String, bytes);
    return (new Function("return " + byteString))();
}

function ConvertToJsonObjectFromString(str)
{
    return (new Function("return " + str))();
}

function isEnableCustomization(scriptContext)
{
    var ret = false;
    try
    {
        ret = scriptContext.QueueProperties.GetBool("JS_Customize_Enabled");
    }
    catch(e)
    {
        ret = false;
    }
    return ret;
}

function createPageWatermark( scriptContext, property)
{
    Debug.write("createPageWatermark() is called!");

    var result = null;
    try
    {
        var wmlist = {};
        wmlist[KEY_TAG]       = TAG_FEATURE;
        wmlist[KEY_ATTRIBUTE] = "ns0000:PageWatermark";
        wmlist[KEY_PICKONE]   = {};

        var WMK_ITEM_FEATURE = "ns0000:WatermarkItem";
        var WMK_ITEM_TYPE    = TYPE_STRING;

        //
        var presets = getWatermarkList( scriptContext, property );
        //

        var scoredList = [];
        for(var i=0; i<presets.length; i++ )
        {
            scoredList[i] = {
                "name":WMK_ITEM_FEATURE,
                "type":WMK_ITEM_TYPE,
                "value":presets[i]
            };
        }
        wmlist[RCF_ITEM_KEY_PICKONE]['ns0000:On'] = scoredList;

        var elementList = createFeatureElement( wmlist, property );

        if( elementList.length == 1 )
        {
            result = elementList[0];
        }
    }
    catch(e)
    {
        result = null;
    }
    return result;
}

function getWatermarkList( scriptContext, property )
{
    var watermarkpresets = {
        "english": ["CONFIDENTIAL","ORIGINAL","DRAFT","URGENT","COPY"],
        "japanese": ["CONFIDENTIAL","ЊO","DRAFT","}","COPY"]
    };

    try
    {
        var presets = [];
        if( global.eventtype == EVENT_DEVMODETOPRINTTICKET )
        {
            var itemString = property.getCurrentValue( "ns0000:WatermarkItem" );
            if( itemString != null && itemString != "" )
            {
                presets.push( itemString );
            }
        }
        else if( global.eventtype == EVENT_PRINTCAPABILITIES )
        {
            if( global.language ) {
                presets = watermarkpresets[global.language];
            }
            if( !presets || presets.length == 0 )
            {
                presets = watermarkpresets["english"];
            }

            var settings = scriptContext.UserProperties.getString("watermark_settings");
            var settingsObj  = ConvertToJsonObjectFromString(settings);

            var itemList = settingsObj['personal_watermarks'];
            for(var i=0; i<itemList.length; i++)
            {
                var itemStr = scriptContext.UserProperties.getString(itemList[i]);
                var itemObj = ConvertToJsonObjectFromString(itemStr);
                presets.push( itemObj['settings']['name'] );
            }
        }
        else
        {
            // no
        }
    }
    catch(e)
    {
        Debug.write("Failed: getWatermarkPresetList");
    }

    return presets;
}

function setCurrentLanguage( printCapabilities )
{
    var langList = { "pTCY":"japanese" };
    var xpath = "psf:Feature[@name='psk:PageMediaSize']/psf:Property[@name='psk:DisplayName']";
    var node = printCapabilities.XmlNode.documentElement.selectSingleNode( xpath );

    global.language = langList[node.text];
}


/**************************************************************
*                                                             *
*              Utility functions                              *
*                                                             *
**************************************************************/

function setPropertyValue(propertyNode, value) {
    //// <summary>
    ////     Set the value contained in the 'Value' node under a 'Property'
    ////     or a 'ScoredProperty' node in the print ticket/print capabilities document.
    //// </summary>
    //// <param name="propertyNode" type="IXMLDOMNode">
    ////     The 'Property'/'ScoredProperty' node.
    //// </param>
    //// <param name="value" type="variant">
    ////     The value to be stored under the 'Value' node.
    //// </param>
    //// <returns type="IXMLDOMNode" mayBeNull="true" locid="R:propertyValue">
    ////     First child 'Property' node if found, Null otherwise.
    //// </returns>
    var valueNode = getPropertyFirstValueNode(propertyNode);
    if (valueNode) {
        var child = valueNode.firstChild;
        if (child) {
            child.nodeValue = value;
            return child;
        }
    }
    return null;
}


function setSubPropertyValue(parentProperty, keywordNamespace, subPropertyName, value) {
    //// <summary>
    ////     Set the value contained in an inner Property node's 'Value' node (i.e. 'Value' node in a Property node
    ////     contained inside another Property node).
    //// </summary>
    //// <param name="parentProperty" type="IXMLDOMNode">
    ////     The parent property node.
    //// </param>
    //// <param name="keywordNamespace" type="String">
    ////     The namespace in which the property name is defined.
    //// </param>
    //// <param name="subPropertyName" type="String">
    ////     The name of the sub-property node.
    //// </param>
    //// <param name="value" type="variant">
    ////     The value to be set in the sub-property node's 'Value' node.
    //// </param>
    //// <returns type="IXMLDOMNode" mayBeNull="true">
    ////     Refer setPropertyValue.
    //// </returns>
    if (!parentProperty ||
        !keywordNamespace ||
        !subPropertyName) {
        return null;
    }
    var subPropertyNode = getProperty(
                            parentProperty,
                            keywordNamespace,
                            subPropertyName);
    return setPropertyValue(
            subPropertyNode,
            value);
}

function getScoredProperty(node, keywordNamespace, scoredPropertyName) {
    //// <summary>
    ////     Retrieve a 'ScoredProperty' element in a print ticket/print capabilities document.
    //// </summary>
    //// <param name="node" type="IXMLDOMNode">
    ////     The scope of the search i.e. the parent node.
    //// </param>
    //// <param name="keywordNamespace" type="String">
    ////     The namespace in which the element's 'name' attribute is defined.
    //// </param>
    //// <param name="scoredPropertyName" type="String">
    ////     The ScoredProperty's 'name' attribute (without the namespace prefix).
    //// </param>
    //// <returns type="IXMLDOMNode" mayBeNull="true">
    ////     The node on success, 'null' on failure.
    //// </returns>

    // Note: It is possible to hard-code the 'psfPrefix' variable in the tag name since the
    // SelectionNamespace property has been set against 'psfPrefix'
    // in validatePrintTicket/completePrintCapabilities.
    return searchByAttributeName(
                node,
                psfPrefix + ":ScoredProperty",
                keywordNamespace,
                scoredPropertyName);
}

function getProperty(node, keywordNamespace, propertyName) {
    //// <summary>
    ////     Retrieve a 'Property' element in a print ticket/print capabilities document.
    //// </summary>
    //// <param name="node" type="IXMLDOMNode">
    ////     The scope of the search i.e. the parent node.
    //// </param>
    //// <param name="keywordNamespace" type="String">
    ////     The namespace in which the element's 'name' attribute is defined.
    //// </param>
    //// <param name="propertyName" type="String">
    ////     The Property's 'name' attribute (without the namespace prefix).
    //// </param>
    //// <returns type="IXMLDOMNode" mayBeNull="true">
    ////     The node on success, 'null' on failure.
    //// </returns>
    return searchByAttributeName(
            node,
            psfPrefix + ":Property",
            keywordNamespace,
            propertyName);
}

function setSelectedOptionName(printSchemaFeature, keywordPrefix, optionName) {
    //// <summary>
    ////      Set the 'name' attribute of a Feature's selected option
    ////      Note: This function should be invoked with Feature type that is retrieved
    ////            via either PrintCapabilties->GetFeature() or PrintTicket->GetFeature().
    ////
    ////      Caution: Setting only the 'name' attribute can result in an invalid option element.
    ////            Some options require their entire subtree to be updated.
    //// </summary>
    //// <param name="printSchemaFeature" type="IPrintSchemaFeature">
    ////     Feature variable.
    //// </param>
    //// <param name="keywordPrefix" type="String">
    ////     The prefix for the optionName parameter.
    //// </param>
    //// <param name="optionName" type="String">
    ////     The name (without prefix) to set as the 'name' attribute.
    //// </param>
    if (!printSchemaFeature ||
        !printSchemaFeature.SelectedOption ||
        !printSchemaFeature.SelectedOption.XmlNode) {
        return;
    }
    printSchemaFeature.SelectedOption.XmlNode.setAttribute(
        "name",
        keywordPrefix + ":" + optionName);
}


/**************************************************************
*                                                             *
*              Functions used by utility functions            *
*                                                             *
**************************************************************/

function getPropertyFirstValueNode(propertyNode) {
    //// <summary>
    ////     Retrieve the first 'value' node found under a 'Property' or 'ScoredProperty' node.
    //// </summary>
    //// <param name="propertyNode" type="IXMLDOMNode">
    ////     The 'Property'/'ScoredProperty' node.
    //// </param>
    //// <returns type="IXMLDOMNode" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!propertyNode) {
        return null;
    }

    var nodeName = propertyNode.nodeName;
    if ((nodeName.indexOf(":Property") < 0) &&
        (nodeName.indexOf(":ScoredProperty") < 0)) {
        return null;
    }

    var valueNode = propertyNode.selectSingleNode(psfPrefix + ":Value");
    return valueNode;
}

function searchByAttributeName(node, tagName, keywordNamespace, nameAttribute) {
    //// <summary>
    ////      Search for a node that with a specific tag name and containing a
    ////      specific 'name' attribute
    ////      e.g. &lt;Bar name=\"ns:Foo\"&gt; is a valid result for the following search:
    ////           Retrieve elements with tagName='Bar' whose nameAttribute='Foo' in
    ////           the namespace corresponding to prefix 'ns'.
    //// </summary>
    //// <param name="node" type="IXMLDOMNode">
    ////     Scope of the search i.e. the parent node.
    //// </param>
    //// <param name="tagName" type="String">
    ////     Restrict the searches to elements with this tag name.
    //// </param>
    //// <param name="keywordNamespace" type="String">
    ////     The namespace in which the element's name is defined.
    //// </param>
    //// <param name="nameAttribute" type="String">
    ////     The 'name' attribute to search for.
    //// </param>
    //// <returns type="IXMLDOMNode" mayBeNull="true">
    ////     IXMLDOMNode on success, 'null' on failure.
    //// </returns>
    if (!node ||
        !tagName ||
        !keywordNamespace ||
        !nameAttribute) {
        return null;
    }

    // For more information on this XPath query, visit:
    // http://blogs.msdn.com/b/benkuhn/archive/2006/05/04/printticket-names-and-xpath.aspx
    var xPathQuery = "descendant::" +
                     tagName +
                     "[substring-after(@name,':')='" +
                     nameAttribute +
                     "']" +
                     "[name(namespace::*[.='" +
                     keywordNamespace +
                     "'])=substring-before(@name,':')]"
    ;

    var nodeRet = node.selectSingleNode(xPathQuery);

    if (nodeRet !== null) {
        return nodeRet;
    }

    return null;
}

function setSelectionNamespace(xmlNode, prefix, namespace) {
    //// <summary>
    ////     This function sets the 'SelectionNamespaces' property on the XML Node.
    ////     For more details: http://msdn.microsoft.com/en-us/library/ms756048(VS.85).aspx
    //// </summary>
    //// <param name="xmlNode" type="IXMLDOMNode">
    ////     The node on which the property is set.
    //// </param>
    //// <param name="prefix" type="String">
    ////     The prefix to be associated with the namespace.
    //// </param>
    //// <param name="namespace" type="String">
    ////     The namespace to be added to SelectionNamespaces.
    //// </param>
    xmlNode.setProperty(
        "SelectionNamespaces",
        "xmlns:" +
            prefix +
            "='" +
            namespace +
            "'"
        );
}


function getPrefixForNamespace(node, namespace) {
    //// <summary>
    ////     This function returns the prefix for a given namespace.
    ////     Example: In 'psf:printTicket', 'psf' is the prefix for the namespace.
    ////     xmlns:psf="http://schemas.microsoft.com/windows/2003/08/printing/printschemaframework"
    //// </summary>
    //// <param name="node" type="IXMLDOMNode">
    ////     A node in the XML document.
    //// </param>
    //// <param name="namespace" type="String">
    ////     The namespace for which prefix is returned.
    //// </param>
    //// <returns type="String">
    ////     Returns the namespace corresponding to the prefix.
    //// </returns>

    if (!node) {
        return null;
    }

    // navigate to the root element of the document.
    var rootNode = node.documentElement;

    // Query to retrieve the list of attribute nodes for the current node
    // that matches the namespace in the 'namespace' variable.
    var xPathQuery = "namespace::node()[.='" +
                namespace +
                "']";
    var namespaceNode = rootNode.selectSingleNode(xPathQuery);
    var prefix = namespaceNode.baseName;

    return prefix;
}


function CreateFeatureSelection(xmlDOMDocInput, strFeatureName, strDisplayName) {
    return CreateFeatureSelection(xmlDOMDocInput, strFeatureName, strDisplayName, pskPrefix);
}

function CreateFeatureSelection(xmlDOMDocInput, strFeatureName, strDisplayName, strPrefix) {
    //// <summary>
    ////     Create a node with name of "strFeatureName", with display name of "strDisplayName"
    //// </summary>
    //// <param name="strFeatureName" type="String">
    ////     The name of the feature node.
    //// <param name="strDisplayName" type="String">
    ////     The display name of the feature node.
    //// </param>
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strFeatureName || strFeatureName.length === 0) {
        return null;
    }

    var featureElement = null;
    featureElement = CreateFeature(xmlDOMDocInput, strFeatureName, strDisplayName, strPrefix);
    if (!featureElement) {
        return null;
    }

    var strPickOne = pskPrefix + ":" + PICKONE_VALUE_NAME;
    var propertyTypeElement = null;
    var strStringType = SCHEMA_XS + ":" + SCHEMA_QNAME;

    propertyTypeElement = CreateFWPropertyWithInput(xmlDOMDocInput, SELECTIONTYPE_VALUE_NAME, strStringType, strPickOne);
    if (propertyTypeElement) {
        featureElement.appendChild(propertyTypeElement);
    }

    return featureElement;
}


function CreateFeature(xmlDOMDocInput, strFeatureNameInput, strDisplayNameInput) {
    return CreateFeature(xmlDOMDocInput, strFeatureNameInput, strDisplayNameInput, pskPrefix);
}


function CreateFeature(xmlDOMDocInput, strFeatureNameInput, strDisplayNameInput, strNSprefix) {
    //// <summary>
    ////     Create a node with name of "strFeatureName", with display name of "strDisplayName"
    //// </summary>
    //// <param name="strFeatureNameInput" type="String">
    ////     The name of the feature node.
    //// <param name="strDisplayNameInput" type="String">
    ////     The display name of the feature node.
    //// </param>
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strFeatureNameInput || strFeatureNameInput.length === 0) {
        return null;
    }

    var strFeature = psfPrefix + ":" + FEATURE_ELEMENT_NAME;
    var strFeatureAttrib = strNSprefix + ":" + strFeatureNameInput;

    var featureElement = null;
    featureElement = CreateXMLElement(xmlDOMDocInput, strFeature, FRAMEWORK_URI);
    if (featureElement) {
        CreateXMLAttribute(xmlDOMDocInput, featureElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strFeatureAttrib);
    }

    if (strDisplayNameInput) {
        var displayProp = CreateFWPropertyWithInput(xmlDOMDocInput, DISPLAYNAME_VALUE_NAME, SCHEMA_STRING, strDisplayNameInput);
        if (displayProp) {
            featureElement.appendChild(displayProp);
        }
    }
    return featureElement;
}


function CreateFWPropertyWithInput(xmlDOMDocInput, strPropNameInput, strTypeInput, strValueInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <param name="strTypeInput" type="String">
    ////     The type of the feature node.
    //// </param>
    //// <param name="strValueInput" type="String">
    ////     The value of the feature node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropNameInput || strPropNameInput.length === 0) {
        return null;
    }

    var propElement = null;

    propElement = CreateFWProperty(xmlDOMDocInput, strPropNameInput);

    if (propElement) {

        var valueElement = null;

        var strValueElement = psfPrefix + ":" + VALUE_ELEMENT_NAME;
        var strTypeAttribName = SCHEMA_XSI + ":" + SCHEMA_TYPE;
        var strTypeAttribValue = strTypeInput;

        valueElement = CreateXMLElement(xmlDOMDocInput, strValueElement, FRAMEWORK_URI);
        if (valueElement) {
            valueElement.text = strValueInput;
            CreateXMLAttribute(xmlDOMDocInput, valueElement, strTypeAttribName, SCHEMA_INST_URI, strTypeAttribValue);
            propElement.appendChild(valueElement);
        }
    }

    return propElement;
}

function CreateFWProperty(xmlDOMDocInput, strPropNameInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropNameInput || strPropNameInput.length === 0) {
        return null;
    }

    var propElement = null;
    var strTagName = psfPrefix + ":" + PROPERTY_ELEMENT_NAME;
    var strAttribName = psfPrefix + ":" + strPropNameInput;

    propElement = CreateXMLElement(xmlDOMDocInput, strTagName, FRAMEWORK_URI);
    if (propElement) {
        CreateXMLAttribute(xmlDOMDocInput, propElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strAttribName);
    }

    return propElement;
}

function CreateScoredPropertyRefElement(xmlDOMDocInput, strPropNameInput, strParamRefNameAttribInput) {
    return CreateScoredPropertyRefElement(xmlDOMDocInput, strPropNameInput, strParamRefNameAttribInput, pskPrefix);
}

function CreateScoredPropertyRefElement(xmlDOMDocInput, strPropNameInput, strParamRefNameAttribInput, strPrefix_input) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropNameInput || strPropNameInput.length === 0) {
        return null;
    }

    if (!strParamRefNameAttribInput || strParamRefNameAttribInput.length === 0) {
        return null;
    }

    var propElement = null;
    var strTagName = psfPrefix + ":" + SCORED_PROP_ELEMENT_NAME;

    var strTagNameUrl = psfPrefix + ":" + psfNs;

    var strAttribName = strPrefix_input + ":" + strPropNameInput;

    propElement = CreateXMLElement(xmlDOMDocInput, strTagName, FRAMEWORK_URI);
    if (propElement) {
        CreateXMLAttribute(xmlDOMDocInput, propElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strAttribName);
    }

    var paramRefElement = null;
    var strParamRefName = psfPrefix + ":" + PARAM_REF_ELEMENT_NAME;
    var strparamAttribName = strPrefix_input + ":" + strParamRefNameAttribInput;

    paramRefElement = CreateXMLElement(xmlDOMDocInput, strParamRefName, FRAMEWORK_URI);
    if (paramRefElement) {
        CreateXMLAttribute(xmlDOMDocInput, paramRefElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strparamAttribName);
    }

    if (propElement && paramRefElement) {
        propElement.appendChild(paramRefElement);
    }

    return propElement;
}


function CreateScoredPropertyIntegerElement(xmlDOMDocInput, strPropNameInput, strParamValue) {
    return CreateScoredPropertyRefElement(xmlDOMDocInput, strPropNameInput, strParamValue, pskPrefix);
}

function CreateScoredPropertyIntegerElement(xmlDOMDocInput, strPropNameInput, strParamValue, strPrefix_input) {
    //// <summary>
    ////     creates a property element of the given integer "strParamValue".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropNameInput || strPropNameInput.length === 0) {
        return null;
    }

    var propElement = null;
    var strTagName = psfPrefix + ":" + SCORED_PROP_ELEMENT_NAME;

    var strTagNameUrl = psfPrefix + ":" + psfNs;

    var strAttribName = strPrefix_input + ":" + strPropNameInput;

    propElement = CreateXMLElement(xmlDOMDocInput, strTagName, FRAMEWORK_URI);
    if (propElement) {
        CreateXMLAttribute(xmlDOMDocInput, propElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strAttribName);
    }

    var strValueElement = psfPrefix + ":" + VALUE_ELEMENT_NAME;
    var strTypeAttribName = SCHEMA_XSI + ":" + SCHEMA_TYPE;
    var strIntegerType = SCHEMA_XS + ":" + SCHEMA_INTEGER;

    var valueElement = CreateXMLElement(xmlDOMDocInput, strValueElement, FRAMEWORK_URI);
    if (valueElement) {
        valueElement.text = strParamValue;
        CreateXMLAttribute(xmlDOMDocInput, valueElement, strTypeAttribName, SCHEMA_INST_URI, strIntegerType);
    }

    if (propElement && valueElement) {
        propElement.appendChild(valueElement);
    }

    return propElement;
}


function CreateScoredPropertyStringElement(xmlDOMDocInput, strPropNameInput, strParamValue) {
    return CreateScoredPropertyStringElement(xmlDOMDocInput, strPropNameInput, strParamValue, pskPrefix);
}

function CreateScoredPropertyStringElement(xmlDOMDocInput, strPropNameInput, strParamValue, strPrefix_input) {
    //// <summary>
    ////     creates a property element of the given string "strParamValue".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropNameInput || strPropNameInput.length === 0) {
        return null;
    }

    var propElement = null;
    var strTagName = psfPrefix + ":" + SCORED_PROP_ELEMENT_NAME;

    var strTagNameUrl = psfPrefix + ":" + psfNs;

    var strAttribName = strPrefix_input + ":" + strPropNameInput;

    propElement = CreateXMLElement(xmlDOMDocInput, strTagName, FRAMEWORK_URI);
    if (propElement) {
        CreateXMLAttribute(xmlDOMDocInput, propElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strAttribName);
    }

    var strValueElement = psfPrefix + ":" + VALUE_ELEMENT_NAME;
    var strTypeAttribName = SCHEMA_XSI + ":" + SCHEMA_TYPE;
    var strIntegerType = SCHEMA_XS + ":" + SCHEMA_STRING;

    var valueElement = CreateXMLElement(xmlDOMDocInput, strValueElement, FRAMEWORK_URI);
    if (valueElement) {
        valueElement.text = strParamValue;
        CreateXMLAttribute(xmlDOMDocInput, valueElement, strTypeAttribName, SCHEMA_INST_URI, strIntegerType);
    }

    if (propElement && valueElement) {
        propElement.appendChild(valueElement);
    }

    return propElement;
}



function CreateStringParamDef(xmlDOMDocInput, strPropDefNameInput, bIsPublicKeywordInput, strDefaultValueInput, intMinLengthInput, intMaxLengthInput, strUnitTypeInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropDefNameInput || strPropDefNameInput.length === 0) {
        return null;
    }   

    if (!xmlDOMDocInput) {
        return null;
    }

    var strparamDefName = psfPrefix + ":" + PARAM_DEF_ELEMENT_NAME;
    var strParamAttrib = null;

    if (bIsPublicKeywordInput === false) {
        strParamAttrib = userpskPrefix + ":";
    }
    else {
        strParamAttrib = pskPrefix + ":";
    }

    strParamAttrib += strPropDefNameInput;

    var propDefElement = null;

    propDefElement = CreateXMLElement(xmlDOMDocInput, strparamDefName, FRAMEWORK_URI);
    if (propDefElement) {
        CreateXMLAttribute(xmlDOMDocInput, propDefElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strParamAttrib);
    }

    if (propDefElement) {

        var nodeDataTypeProp = null;
        var nodeDefValProp = null;
        var nodeMaxLengthProp = null;
        var nodeMinLengthProp = null;
        var nodeMandatoryProp = null;
        var nodeUnitTypeProp = null;

        var strQNameType = SCHEMA_XS + ":" + SCHEMA_QNAME;
        var strStringType = SCHEMA_XS + ":" + SCHEMA_STRING;
        var strDataType = DATATYPE_VALUE_NAME;
        var strDefValue = DEFAULTVAL_VALUE_NAME;
        var strMaxLength = MAX_LENGTH_NAME;
        var strIntegerType = SCHEMA_XS + ":" + SCHEMA_INTEGER;
        var strMinLength = MIN_LENGTH_NAME;
        var strMandatory = MANDATORY_VALUE_NAME;
        var strMandatoryValue = pskPrefix + ":" + SCHEMA_CONDITIONAL;
        var strUnitType = UNITTYPE_VALUE_NAME;

        nodeDataTypeProp = CreateFWPropertyWithInput(xmlDOMDocInput, strDataType, strQNameType, strStringType);
        if (nodeDataTypeProp) {
            propDefElement.appendChild(nodeDataTypeProp);
        }

        nodeDefValProp = CreateFWPropertyWithInput(xmlDOMDocInput, strDefValue, strStringType, strDefaultValueInput);
        if (nodeDefValProp) {
            propDefElement.appendChild(nodeDefValProp);
        }

        nodeMaxLengthProp = CreateFWPropertyWithInput(xmlDOMDocInput, strMaxLength, strIntegerType, intMaxLengthInput.toString());
        if (nodeMaxLengthProp) {
            propDefElement.appendChild(nodeMaxLengthProp);
        }

        nodeMinLengthProp = CreateFWPropertyWithInput(xmlDOMDocInput, strMinLength, strIntegerType, intMinLengthInput.toString());
        if (nodeMinLengthProp) {
            propDefElement.appendChild(nodeMinLengthProp);
        }

        nodeMandatoryProp = CreateFWPropertyWithInput(xmlDOMDocInput, strMandatory, strQNameType, strMandatoryValue);
        if (nodeMandatoryProp) {
            propDefElement.appendChild(nodeMandatoryProp);
        }

        nodeUnitTypeProp = CreateFWPropertyWithInput(xmlDOMDocInput, strUnitType, strStringType, strUnitTypeInput);
        if (nodeUnitTypeProp) {
            propDefElement.appendChild(nodeUnitTypeProp);
        }

    }

    return propDefElement;
}

function CreateIntParamDef(xmlDOMDocInput, strPropDefNameInput, bIsPublicKeywordInput, intDefaultValueInput, intMinValueInput, intMaxValueInput, intMultipleInout, strUnitTypeInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropDefNameInput || strPropDefNameInput.length === 0) {
        return null;
    }

    if (!xmlDOMDocInput) {
        return null;
    }

    var strparamDefName = psfPrefix + ":" + PARAM_DEF_ELEMENT_NAME;
    var strParamAttrib = null;

    if (bIsPublicKeywordInput === false) {
        strParamAttrib = userpskPrefix + ":";
    }
    else {
        strParamAttrib = pskPrefix + ":";
    }

    strParamAttrib += strPropDefNameInput;
    var propDefElement = null;

    propDefElement = CreateXMLElement(xmlDOMDocInput, strparamDefName, FRAMEWORK_URI);
    if (propDefElement) {
        CreateXMLAttribute(xmlDOMDocInput, propDefElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strParamAttrib);
    }

    if (propDefElement) {

        var nodeDataTypeProp = null;
        var nodeDefValProp = null;
        var nodeMaxValueProp = null;
        var nodeMinValueProp = null;
        var nodeMultipleProp = null;
        var nodeMandatoryProp = null;
        var nodeUnitTypeProp = null;

        var strQNameType = SCHEMA_XS + ":" + SCHEMA_QNAME;
        var strStringType = SCHEMA_XS + ":" + SCHEMA_STRING;
        var strDataType = DATATYPE_VALUE_NAME;
        var strDefValue = DEFAULTVAL_VALUE_NAME;
        var strMaxValue = MAX_VALUE_NAME;
        var strIntegerType = SCHEMA_XS + ":" + SCHEMA_INTEGER;
        var strMinValue = MIN_VALUE_NAME;
        var strMultiple = MULTIPLE_VALUE_NAME;
        var strMandatory = MANDATORY_VALUE_NAME;
        var strMandatoryValue = pskPrefix + ":" + SCHEMA_CONDITIONAL;
        var strUnitType = UNITTYPE_VALUE_NAME;        

        nodeDataTypeProp = CreateFWPropertyWithInput(xmlDOMDocInput, strDataType, strQNameType, strIntegerType);
        if (nodeDataTypeProp) {
            propDefElement.appendChild(nodeDataTypeProp);
        }

        nodeDefValProp = CreateFWPropertyWithInput(xmlDOMDocInput, strDefValue, strIntegerType, intDefaultValueInput.toString());
        if (nodeDefValProp) {
            propDefElement.appendChild(nodeDefValProp);
        }

        nodeMaxValueProp = CreateFWPropertyWithInput(xmlDOMDocInput, strMaxValue, strIntegerType, intMaxValueInput.toString());
        if (nodeMaxValueProp) {
            propDefElement.appendChild(nodeMaxValueProp);
        }

        nodeMinValueProp = CreateFWPropertyWithInput(xmlDOMDocInput, strMinValue, strIntegerType, intMinValueInput.toString());
        if (nodeMinValueProp) {
            propDefElement.appendChild(nodeMinValueProp);
        }

        nodeMultipleProp = CreateFWPropertyWithInput(xmlDOMDocInput, strMultiple, strIntegerType, intMultipleInout.toString());
        if (nodeMultipleProp) {
            propDefElement.appendChild(nodeMultipleProp);
        }

        nodeMandatoryProp = CreateFWPropertyWithInput(xmlDOMDocInput, strMandatory, strQNameType, strMandatoryValue);
        if (nodeMandatoryProp) {
            propDefElement.appendChild(nodeMandatoryProp);
        }

        nodeUnitTypeProp = CreateFWPropertyWithInput(xmlDOMDocInput, strUnitType, strStringType, strUnitTypeInput);
        if (nodeUnitTypeProp) {
            propDefElement.appendChild(nodeUnitTypeProp);
        }

    }

    return propDefElement;
}

function CreateXMLElement(xmlDOMDocInput, strElementNameInput, strTargetURIInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strElementNameInput" type="String">
    ////     The name of the element node.
    //// <param name="strTargetURIInput" type="String">
    ////     The name of the Target URI node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>

    //if (!strElementNameInput || strElementNameInput.length === 0) {
    if (null === strElementNameInput) {
        return null;
    }

    var currentNode = null;
    currentNode = xmlDOMDocInput.XmlNode.createNode(1, strElementNameInput, strTargetURIInput);

    if (currentNode) {
        return currentNode;
    }

    return null;

}

function CreateXMLAttribute(xmlDOMDocInput, elementInput, strNameInput, strTargetURIInput, strValueInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="elementInput" type="IXMLDOMElement">
    ////     The element node.
    //// <param name="strNameInput" type="String">
    ////     The name of the Attribute node.
    //// <param name="strTargetURIInput" type="String">
    ////     The name of the Target URI node.
    //// <param name="strValueInput" type="String">
    ////     The value of the Attribute node.

    if (!strNameInput || strNameInput.length === 0) {
        return null;
    }

    var currentNode = null;
    var currentAttributeNode = null;

    var nodeType = 2;
    currentNode = xmlDOMDocInput.XmlNode.createNode(nodeType, strNameInput, strTargetURIInput);
    
    if (currentNode) {
        if (strValueInput) {
            currentNode.value = strValueInput;
            elementInput.setAttributeNode(currentNode);
        }
    }

}



function CreateOption(xmlDOMDocInput, strOptionNameInput, strDisplayNameInput) {

    return CreateOption(xmlDOMDocInput, strOptionNameInput, strDisplayNameInput, pskPrefix);
}

function CreateOption(xmlDOMDocInput, strOptionNameInput, strDisplayNameInput, strNameSpace) {
    //// <summary>
    ////     Create a node with name of "strFeatureName", with display name of "strDisplayName"
    //// </summary>
    //// <param name="strOptionNameInput" type="String">
    ////     The name of the option node.
    //// <param name="strDisplayNameInput" type="String">
    ////     The display name of the feature node.
    //// </param>
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>

    if (null === strOptionNameInput) {
        return null;
    }

    var strOption = psfPrefix + ":" + OPTION_ELEMENT_NAME;
    var strOptionAttrib = (strOptionNameInput === "") ? strOptionNameInput : strNameSpace + ":" + strOptionNameInput;

    var pOptionElement = null;
    pOptionElement = CreateXMLElement(xmlDOMDocInput, strOption, FRAMEWORK_URI);
    if (pOptionElement) {
        CreateXMLAttribute(xmlDOMDocInput, pOptionElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strOptionAttrib);
    }

    if (strDisplayNameInput) {
        var pDisplayPropElem = CreateFWPropertyWithInput(xmlDOMDocInput, DISPLAYNAME_VALUE_NAME, SCHEMA_STRING, strDisplayNameInput);
        if (pOptionElement && pDisplayPropElem) {
            pOptionElement.appendChild(pDisplayPropElem);
        }
    }

    return pOptionElement;

}

function CreateIntParamRefElement(devModeProperties, printTicket, scriptContext, strParamRefName) {

    var m_pRootDocument = printTicket;
    var strTypeName = null;
    var strValue = null;

    strTypeName = SCHEMA_XS + ":" + SCHEMA_STRING;
    try {
        strValue = devModeProperties.GetString(strParamRefName);
        var nodeParamIniValue = getParamDefInitNode(m_pRootDocument.XmlNode.documentElement, false, strParamRefName);
        if (!nodeParamIniValue ) {
            nodeParamIniValue = CreateIntParamRefIni(m_pRootDocument, strParamRefName, false, strTypeName, strValue.toString());
            if (nodeParamIniValue) {
                printTicket.XmlNode.lastChild.appendChild(nodeParamIniValue);
            }
        }
        else {
            nodeParamIniValue.firstChild.text = strValue.toString();
        }
    }
    catch (e) {
        var defaultValueKey = "Default" + strParamRefName + "Value";
        strValue = scriptContext.DriverProperties.GetString(defaultValueKey);
        var nodeParamIniValue = CreateIntParamRefIni(m_pRootDocument, strParamRefName, false, strTypeName, strValue.toString);
        if (nodeParamIniValue) {
            printTicket.XmlNode.lastChild.appendChild(nodeParamIniValue);
        }
    }
 
}


function getParamDefInitNode(xmlSrcNode_Input, boolIsPublicKeyword, strOptionName_Input) {

    if (!xmlSrcNode_Input) {
        return null;
    }

    if (!strOptionName_Input || strOptionName_Input.length === 0) {
        return null;
    }

    var strNameSpace = null;

    if (boolIsPublicKeyword === false) {
        strNameSpace = ricohNs;
    }
    else {
        strNameSpace = pskNs;
    }

    var xmlNode_Out = searchByAttributeName(xmlSrcNode_Input, psfPrefix + ":" + PARAM_INIT_ELEMENT_NAME, strNameSpace, strOptionName_Input);

    return xmlNode_Out;
}


function CreateIntParamRefIni(xmlDOMDocInput, strPropDefNameInput, bIsPublicKeywordInput, strTypeInput, strValueInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropDefNameInput || strPropDefNameInput.length === 0) {
        return null;
    }

    if (!xmlDOMDocInput) {
        return null;
    }

    var strparamDefName = psfPrefix + ":" + PARAM_INIT_ELEMENT_NAME;
    var strParamAttrib = null;

    if (bIsPublicKeywordInput === false) {
        strParamAttrib = userpskPrefix + ":";
    } else {
        strParamAttrib = pskPrefix + ":";
    }

    strParamAttrib += strPropDefNameInput;

    var propDefElement = null;

    propDefElement = CreateXMLElement(xmlDOMDocInput, strparamDefName, FRAMEWORK_URI);
    if (propDefElement) {
        CreateXMLAttribute(xmlDOMDocInput, propDefElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strParamAttrib);
    }

    if (propDefElement) {

        var valueElement = null;

        var strValueElement = psfPrefix + ":" + VALUE_ELEMENT_NAME;
        var strTypeAttribName = SCHEMA_XSI + ":" + SCHEMA_TYPE;
        var strAttribValue = strTypeInput;

        valueElement = CreateXMLElement(xmlDOMDocInput, strValueElement, FRAMEWORK_URI);
        if (valueElement) {
            valueElement.text = strValueInput;
            CreateXMLAttribute(xmlDOMDocInput, valueElement, strTypeAttribName, SCHEMA_INST_URI, strAttribValue);
            propDefElement.appendChild(valueElement);
        }
    }

    return propDefElement;
}


function GetUnicodeToObject(drvProperties, loadFileName) {
    var bytesData = GetBytesFromProperties(drvProperties, loadFileName);
    var multiChar = WideCharBytesToMultiCharByte(bytesData);
    var object = ConvertToJsonObjectFromBytes(multiChar);

    return object;
}

function CreatelangPropertyWithInput(xmlDOMDocInput, strPropNameInput, strTypeInput, strValueInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <param name="strTypeInput" type="String">
    ////     The type of the feature node.
    //// </param>
    //// <param name="strValueInput" type="String">
    ////     The value of the feature node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropNameInput || strPropNameInput.length === 0) {
        return null;
    }

    var propElement = null;

    propElement = CreatelangProperty(xmlDOMDocInput, strPropNameInput);

    if (propElement) {

        var valueElement = null;

        var strValueElement = psfPrefix + ":" + VALUE_ELEMENT_NAME;
        var strTypeAttribName = SCHEMA_XSI + ":" + SCHEMA_TYPE;
        var strTypeAttribValue = SCHEMA_XS + ":" + strTypeInput;

        valueElement = CreateXMLElement(xmlDOMDocInput, strValueElement, FRAMEWORK_URI);
        if (valueElement) {
            valueElement.text = strValueInput;
            CreateXMLAttribute(xmlDOMDocInput, valueElement, strTypeAttribName, SCHEMA_INST_URI, strTypeAttribValue);
            propElement.appendChild(valueElement);
        }
    }

    return propElement;
}

function CreatelangProperty(xmlDOMDocInput, strPropNameInput) {
    //// <summary>
    ////     creates a property element of the given name "strPropNameInput".
    //// </summary>
    //// <param name="strPropNameInput" type="String">
    ////     The name of the property node.
    //// <returns type="IXMLDOMElement" mayBeNull="true">
    ////     The 'Value' node on success, 'null' on failure.
    //// </returns>
    if (!strPropNameInput || strPropNameInput.length === 0) {
        return null;
    }

    var propElement = null;
    var strTagName = psfPrefix + ":" + PROPERTY_ELEMENT_NAME;
    var strAttribName = pskPrefix + ":" + strPropNameInput;

    propElement = CreateXMLElement(xmlDOMDocInput, strTagName, FRAMEWORK_URI);
    if (propElement) {
        CreateXMLAttribute(xmlDOMDocInput, propElement, NAME_ATTRIBUTE_NAME, FRAMEWORK_URI, strAttribName);
    }

    return propElement;
}


